<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\User\CreateUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\MediaHandler;
use App\Models\Round;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    use MediaHandler;

    public function index(): AnonymousResourceCollection
    {
        $admins = User::Admins()
            ->with('role.permissions')
            ->useFilters()
            ->latest()
            ->dynamicPaginate(50);
        return UserResource::collection($admins);
    }

    public function store(CreateUserRequest $request): JsonResponse
    {
        $data = $request->validated();

        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::upload($request->file('photo'), 'admin/photos')
            : null;

        $data['type'] = UserType::ADMIN;
        $admin = User::create($data);

        if (!empty($data['assign_round'])) {
            $rounds = in_array('all', $data['assign_round'])
                ? Round::pluck('id')->toArray()
                : $data['assign_round'];

            $admin->assign_round = array_values(array_unique($rounds));
            $admin->save();
        }

        $admin->load('role.permissions');
        return $this->apiResponseStored(new UserResource($admin));
    }

    public function show(User $admin): JsonResponse
    {
        if (!$admin || $admin->type !== UserType::ADMIN->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }

        $admin->load('role.permissions');
        return $this->apiResponseShow(new UserResource($admin));
    }

    public function update(UpdateUserRequest $request, User $admin): JsonResponse
    {
        if (!$admin || $admin->type !== UserType::ADMIN->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        $data = $request->validated();

        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::updateMedia($request->file('photo'), 'admin/photos', $admin->photo_path)
            : $admin->photo;

        $admin->update($data);

        if (!empty($data['assign_round'])) {
            $rounds = in_array('all', $data['assign_round'])
                ? Round::pluck('id')->toArray()
                : $data['assign_round'];

            $admin->assign_round = array_values(array_unique($rounds));
            $admin->save();
        }


        $admin->load('role.permissions');
        return $this->apiResponseUpdated(new UserResource($admin));
    }

    public function destroy(User $admin): JsonResponse
    {
        if (!$admin || $admin->type !== UserType::ADMIN->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }

        $admin->photo ? MediaHandler::deleteMedia($admin->photo_path) : null;
        $admin->role()->dissociate();
        $admin->delete();
        return $this->apiResponseDeleted();
    }
}
