<?php

namespace App\Http\Controllers\Api\V1\ClientSide;

use App\Enums\DefineStatus;
use App\Http\Traits\NotificationTrait;
use App\Models\User;
use App\Enums\UserType;
use App\Services\OtpService;
use Illuminate\Http\Request;
use App\Http\Controllers\Controller;
use App\Http\Resources\User\UserResource;
use App\Http\Requests\Auth\AuthWithPhoneRequest;
use App\Http\Requests\Auth\AuthWithPhoneAndNameRequest;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;

class AuthController extends Controller
{
    use NotificationTrait;
    public function __construct(protected OtpService $otpService)
    {
        //
    }


    public function register(AuthWithPhoneAndNameRequest $request)
    {
        $data = $request->validated();

        $user = User::firstOrCreate(
            ['phone' => $data['phone']],
            [
                'phone' => $data['phone'],
                'name'  => $data['name'],
                'type'  => UserType::STUDENT->value
            ]
        );

        $this->sendPushNotificationClientSide('client_side', 'register');

        $this->otpService->generateOtp($user->phone);

        return $this->apiResponse(
            ['phone' => $user->phone],
            __('main.otp.otp_generated_successfully'),
            200
        );
    }

    public function loginWithPhone(AuthWithPhoneRequest $request)
    {
        $data = $request->validated();

        // $user = User::where('phone',$data['phone'])->firstOrFail();

        // if (!$user){
        //     return $this->apiResponseFailed(__('main.user_not_found'));
        // }

        $credentials = $request->only('email', 'password');

        if (Auth::attempt($credentials)) {
            $user = Auth::user();

            if ($user->status == DefineStatus::INACTIVE->value) {
                return $this->apiResponseFailed(__('main.user_inactive'));
            }

            // $this->otpService->generateOtp($user->phone);

            // return $this->apiResponse(
            //     ['phone' => $user->phone],
            //     __('main.otp.otp_generated_successfully'),
            //     200
            // );
            $token = $user->createToken('auth_token')->plainTextToken;
            return $this->apiResponse([
                'token' => $token,
                'user'  => new UserResource($user),
            ], __('main.logged_in'), 200);
        }

        return $this->apiResponseFailed(__('main.invalid_credentials'));
    }

    /**
     * Verify OTP and login user
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function verifyOtp(Request $request)
    {
        // Log::info('OTP verification initiated', ['request' => $request->all()]);

        $data = $request->validate([
            'phone' => ['required'],
            'otp'   => ['required', 'string'],
            'device_token' => ['required'],
        ]);

        // Log::debug('Validated data', $data);

        $user = User::where('phone', $data['phone'])->firstOrFail();
        // Log::info('User found', ['user_id' => $user->id]);

        $verificationResult = $this->otpService->verifyOtp($user->phone, $data['otp']);
        // Log::info('OTP verification result', ['status' => $verificationResult->status]);

        if (env('APP_ENV') == 'local') {
            $user->update(['device_token' => $data['device_token']]);
            $token = $user->createToken('auth_token')->plainTextToken;
            Log::info('Token generated in local environment', ['token' => $token]);

            return $this->apiResponse([
                'token' => $token,
                'user'  => new UserResource($user),
            ], __('main.logged_in'), 200);
        }

        if (!$verificationResult->status) {
            // Log::warning('OTP verification failed', ['phone' => $data['phone'], 'message' => $verificationResult->message]);

            return $this->apiResponse(null, $verificationResult->message, 422);
        }

        $user->update(['device_token' => $data['device_token']]);
        $user->save();
        // Log::info('Device token updated', ['user' => $user, 'device_token' => $data['device_token']]);

        $token = $user->createToken('auth_token')->plainTextToken;
        // Log::info('Token generated after successful OTP verification', ['user_id' => $user]);

        return $this->apiResponse([
            'token' => $token,
            'user'  => new UserResource($user),
        ], __('main.logged_in'), 200);
    }


    /**
     * Logout user
     *
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function logout(Request $request)
    {
        $user=$request->user();
        $user->device_token=null;
        $user->save();
        $user->currentAccessToken()->delete();
        return $this->apiResponse(null, __('main.logged_out'), 200);
    }
}
