<?php

namespace App\Http\Controllers\Api\V1\ClientSide;

use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentStatus;
use App\Enums\TransactionTypeEnum;
use App\Enums\TypeCoupon;
use App\Http\Controllers\Controller;
use App\Http\Requests\CourseEnrollment\CreateCourseEnrollmentRequest;
use App\Models\Coupon;
use App\Models\Course;
use App\Models\CourseEnrollment;
use App\Models\Round;
use App\Models\RoundStudent;
use App\Models\Transaction;
use App\Models\User;
use App\Notifications\InvoicePaidMail;
use App\Services\GenerateInvoice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;

class FawryController extends Controller
{
    public function pay(CreateCourseEnrollmentRequest $rquest){

        $data=$rquest->validated();
        $userId = request()->user()->id;
        $course = Course::findOrFail($data['course_id']);

        $this->validatePreviousEnrollment($userId, $data['course_id']);
        if (!empty($data['coupon_code'])) {
            $data = $this->applyCoupon($data, $course);
        } else {
            $data['total_amount'] = $course->price;
        }
        $enrollment = $this->createEnrollment($data);

        return response()->json([
            'status' => true,
            'url' => route('fawry.mobile.payment.page', [
                'enrollment_id' => $enrollment->id,
                'user_id' => $userId,
            ]),
        ]);
    }
    public function payUrl(Request $request)
    {
        $enrollmentId = $request->query('enrollment_id');
        $userId = $request->query('user_id');

        $courseEnrollment = CourseEnrollment::findOrFail($enrollmentId);
        $user = User::findOrFail($userId);

        return view('pay', [
            'user' => $user,
            'enrollment' => $courseEnrollment,
        ]);
    }


    private function validatePreviousEnrollment(int $userId, int $courseId): void
    {
        $previousEnrollment = CourseEnrollment::where('user_id', $userId)
            ->where('course_id', $courseId)
            ->first();

        if (!$previousEnrollment) {
            return;
        }

        if ($this->shouldDeleteUnpaidEnrollment($previousEnrollment)) {
            $previousEnrollment->delete();
            return;
        }

        if ($this->isFullyEnrolled($previousEnrollment)) {
            $this->cleanupActiveRoundsForUser($userId, $courseId);
        }
    }

    private function shouldDeleteUnpaidEnrollment(CourseEnrollment $enrollment): bool
    {
        return $enrollment->paid !=3;
    }

    private function isFullyEnrolled(CourseEnrollment $enrollment): bool
    {
        return $enrollment->paid == 3;
    }

    private function cleanupActiveRoundsForUser(int $userId, int $courseId): void
    {
        $activeRoundIds = Round::query()
            ->forCourse($courseId)
            ->active()
            ->currentlyRunning()
            ->pluck('id');

        RoundStudent::query()
            ->forUser($userId)
            ->forRounds($activeRoundIds)
            ->activeAndPaid()
            ->delete();
    }

    private function applyCoupon(array $data, Course $course): array
    {
        $coupon = Coupon::where('code', $data['coupon_code'])
            ->where('status', true)
            ->where('end_date', '>', now())
            ->whereRaw('uses < max_uses')
            ->first();

        if (!$coupon) {
            throw new \Exception(__('main.invalid_coupon'));
        }

        $type = TypeCoupon::from($coupon->type);

        $discountAmount = match ($type) {
            TypeCoupon::DISCOUNT => ($course->price * $coupon->discount_percentage) / 100,
            TypeCoupon::FIXED => $coupon->discount_amount,
        };

        $data['total_amount'] = max($course->price - $discountAmount, 0);
        $data['coupon_id'] = $coupon->id;

        $coupon->increment('uses');

        return $data;
    }

    private function createEnrollment(array $data): CourseEnrollment
    {
        return CourseEnrollment::create($data);
    }

    public function response(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'merchantRefNum' => 'required|string',
            'fawryRefNumber' => 'required|string',
            'paymentAmount' => 'required|numeric',
            'paymentStatus' => 'required|string|in:PAID,UNPAID,EXPIRED,FAILED',
            'signature' => 'required|string'
        ]);

        if ($validator->fails()) {
            Log::warning('Invalid Fawry callback', $validator->errors());
            return response()->json(['message' => 'Invalid request'], 400);
        }

        $validSignature = hash('sha256',
            $request->merchantRefNum .
            $request->fawryRefNumber .
            number_format($request->paymentAmount, 2, '.', '') .
            config('services.fawry.secure_key')
        );

        if ($request->signature !== $validSignature) {
            Log::error('Invalid Fawry signature', [
                'received' => $request->signature,
                'expected' => $validSignature
            ]);
            return response()->json(['message' => 'Unauthorized'], 403);
        }

        $enrollmentId = $request->input('merchantRefNum');

        $courseEnrollment = CourseEnrollment::where('transaction_id', $enrollmentId)->first();

        if (!$courseEnrollment) {
            return response()->json(['message' => 'Enrollment not found'], 404);
        }

        $courseEnrollment->update([
            'payment_status' => PaymentStatus::PAID->value,
            'payment_type'   => PaymentMethodEnum::FAWRY->value,
            'status'         => true,
            'fawry_ref'      => $request->input('fawryRefNumber'),
            'fawry_response' => json_encode($request->all()),
            'paid'           => 1
        ]);

        $this->createTransaction($courseEnrollment);

        try {
            $invoicePath = $this->generateCourseInvoice($courseEnrollment);

            $notificationData = [
                'title'   => 'Your Course Payment Receipt',
                'message' => 'Your payment for "' . $courseEnrollment->course->title . '" was successful.',
                'subject' => 'Payment Confirmation: ' . $courseEnrollment->course->title,
            ];

            $user = $courseEnrollment->user;
            if ($user) {
                $user->notify(new InvoicePaidMail($invoicePath, $notificationData));
            }

            return response()->json(['message' => 'Payment processed and invoice sent.'], 200);

        } catch (\Exception $e) {
            Log::error('Payment success, but invoice handling failed.', [
                'course_enrollment_id' => $courseEnrollment->id,
                'error' => $e->getMessage(),
            ]);

            return response()->json(['message' => 'Payment saved but invoice generation failed.'], 500);
        }
    }


    protected function generateCourseInvoice(CourseEnrollment $enrollment): string
    {
        $course = $enrollment->course;
        $user = $enrollment->user;
        $amount = $enrollment->total_amount / 100;

        $invoiceService = new GenerateInvoice();

        return $invoiceService
            ->setCustomer([
                'name' => $user->name,
                'address' => $user->address ?? 'N/A',
                'email' => $user->email,
                'phone' => $user->phone ?? 'N/A',
            ])
            ->setSeller([
                'name' => config('app.name'),
                'address' => config('app.address'),
                'email' => config('app.email'),
                'phone' => config('app.phone'),
            ])
            ->addItem(
                $course->title,
                $amount,
                1
            )
            ->setLogo(config('app.logo'))
            ->setInvoiceNumber($enrollment->id)
            ->saveToDisk('public', 'invoices')
            ->save();
    }
    public function createTransaction(CourseEnrollment $enrollment){
        Transaction::create([
            'user_id'         => $enrollment->user_id,
            'course_enrollment_id' => $enrollment->id,
            'amount'          =>$enrollment->total_amount,
            'payment_method'  => PaymentMethodEnum::FAWRY->value,
            'payment_date'    => now(),
            'type'            => TransactionTypeEnum::INCOME,
            'notes'           => 'Auto transaction from fawry payment',
            'created_by'      => $enrollment->user_id,
            'updated_by'      => $enrollment->user_id,
        ]);
    }
}
