<?php

namespace App\Http\Controllers\Api\V1\ClientSide;

use App\Enums\QuestionType;
use App\Http\Controllers\Controller;
use App\Http\Resources\Course\CourseResource;
use App\Http\Resources\PlacementTestClientSide\CategoryResource;
use App\Http\Resources\PlacementTestClientSide\PlacementTestResultResource;
use App\Http\Resources\PlacementTestClientSide\QuestionResource;
use App\Http\Traits\ApiResponse;
use App\Http\Traits\NotificationTrait;
use App\Models\BankOption;
use App\Models\Category;
use App\Models\Course;
use App\Models\PlacementTestResult;
use App\Models\Question;
use App\Models\Setting;
use Hamcrest\Core\Set;
use Illuminate\Http\Request;

class PlacementTestController extends Controller
{
    use ApiResponse, NotificationTrait;
    public function index()
    {
        if (!auth()->user()->placement_test_status) {
            return $this->apiResponse([]);
        }

        $requiredKeys = ['listening', 'reading', 'case', 'time_of_exam'];
        $settingsFromDb = Setting::whereIn('key', $requiredKeys)->pluck('value', 'key');

        $settings = [
            'listening'     => (string) ($settingsFromDb['listening'] ?? 1),
            'reading'       => (string) ($settingsFromDb['reading'] ?? 8),
            'case'          => (string) ($settingsFromDb['case'] ?? 1),
            'time_of_exam'  => (string) ($settingsFromDb['time_of_exam'] ?? 60),
        ];

        $categories = Category::orderBy('order')->get();
        $resultQuestions = collect();

        foreach ($categories as $category) {
            $readingQuestions = $this->getRandomQuestions($category->id, QuestionType::Reading->value, $settings['reading']);

            $listeningQuestions = $this->getRandomQuestions($category->id, QuestionType::Listening->value, $settings['listening']);

            $caseQuestions = $this->getRandomQuestions($category->id, QuestionType::Case->value, $settings['case']);

            $categoryQuestions = $readingQuestions
                ->merge($listeningQuestions)
                ->merge($caseQuestions);

            $resultQuestions = $resultQuestions->merge($categoryQuestions);
        }

        return $this->apiResponse([
            ...$settings,
            'questions_count'=>$resultQuestions->count(),
            'questions' => QuestionResource::collection($resultQuestions),
        ]);
    }

    private function getRandomQuestions($categoryId, $type, $limit)
    {
        return Question::with(['category', 'options', 'tag'])
            ->where('category_id', $categoryId)
            ->where('question_type', $type)
            ->inRandomOrder()
            ->limit($limit)
            ->get();
    }


    public function results(){
        $placement_test_result=PlacementTestResult::where('user_id',auth()->user()->id)->useFilters()->dynamicPaginate();
        return PlacementTestResultResource::collection($placement_test_result);
    }

    public function checkAnswers(Request $request){
        $levelResults = [
            'percentage'=>0,
            'total_questions'=>0,
            'total_correct_answers'=>0,
            'max_level'=>null,
            'avaleble_courses'=>[],
            'levels'=>[]
        ];

        $total_correct_answers=0;
        $total_questions=0;

        foreach($request->all() as $question){
            $levelId = $question['category']['id'];
            $levelName = $question['category']['name'];

            if(!isset($levelResults['levels'][$levelId])){
                $levelResults['levels'][$levelId] = [
                    'level_id' => $levelId,
                    'level' => $levelName,
                    'correct_answer_count' => 0,
                    'total_questions' => 0
                ];
            }

            $questionCorrectCount = 0;
            foreach($question['options'] as $option){
                if(BankOption::where('id',$option['id'])->where('is_correct',1)->exists() && $option['checked']){
                    $questionCorrectCount++;
                    $total_correct_answers++;
                }
            }

            $levelResults['levels'][$levelId]['correct_answer_count'] += $questionCorrectCount;
            $levelResults['levels'][$levelId]['total_questions']++;

            $total_questions++;
        }

        $levelResults['total_correct_answers']=$total_correct_answers;
        $levelResults['total_questions']=$total_questions;

        $levelResults['levels'] = array_values($levelResults['levels']);

        if ($total_correct_answers == 0) {
            return $this->apiResponse([], __('main.placement_test_failed'), 422);
        }

        $levelResults['percentage']=$this->percentage($levelResults['total_correct_answers'],$levelResults['total_questions']);
        $levelResults['max_level']=$this->maxLevel($levelResults['levels']);
        $levelResults['avaleble_courses']=$this->avalebleCourses($levelResults['max_level']['id']);

        PlacementTestResult::create([
            'user_id'=>auth()->user()->id,
            'total_correct_answers'=>$levelResults['total_correct_answers'],
            'total_questions'=>$levelResults['total_questions'],
            'max_level_id'=>$levelResults['max_level']['id'],
            'percentage'=>$levelResults['percentage'],
            'levels'=>$levelResults['levels']
        ]);

        $this->sendPushNotificationClientSide('client_side', 'placement_test');

        return $this->apiResponse($levelResults);
    }


    public function percentage($total_correct_answers,$total_questions){
        $percentage = ($total_correct_answers / $total_questions) * 100;
        return $percentage;
    }
    public function maxLevel($levelResults){
        if (empty($levelResults)) {
            return [
                'id' => null,
                'name' => 'No Level',
                'correct_answer_count' => 0,
                'total_questions' => 0
            ];
        }

        $maxLevel = null;
        $maxCorrectAnswers = 0;
        foreach($levelResults as $level){
            if($level['correct_answer_count'] > $maxCorrectAnswers){
                $maxCorrectAnswers = $level['correct_answer_count'];
                $maxLevel = [
                    'id' => $level['level_id'],
                    'name' => $level['level'],
                    'correct_answer_count' => $level['correct_answer_count'],
                    'total_questions' => $level['total_questions']
                ];
            }
        }

        if ($maxLevel === null) {
            $maxLevel = [
                'id' => $levelResults[0]['level_id'],
                'name' => $levelResults[0]['level'],
                'correct_answer_count' => 0,
                'total_questions' => $levelResults[0]['total_questions']
            ];
        }

        return $maxLevel;
    }
    public function avalebleCourses($id){
        $avalebleCourses = Course::where('category_id',$id)->where('placement_test_available','1')->where('status',1)->with('lectures','category','rounds')->get();
        return CourseResource::collection($avalebleCourses);
    }
}
