<?php


namespace App\Http\Controllers\Api\V1;
use App\Http\Controllers\Controller;
use App\Models\Expense;
use App\Http\Resources\Expense\ExpenseResource;
use Illuminate\Http\Request;
use App\Models\PaymentMethod;
class ExpenseController extends Controller
{
    public function index()
    {
        $expenses = Expense::with(['expenseCategory', 'acceptedBy','paymentMethod'])
            ->latest()
            ->dynamicPaginate();
        return ExpenseResource::collection($expenses);
    }
    public function show(Expense $expense)
    {
        $expense->load(['expenseCategory', 'acceptedBy','paymentMethod']);
        return $this->apiResponseShow(new ExpenseResource($expense));
    }
    public function store(Request $request)
    {
        $data = $request->validate([
            'expense_category_id' => 'required|exists:expense_categories,id',
            'bio' => 'nullable|string',
            'cost' => 'required|numeric',
            'accepted_by' => 'nullable|exists:users,id',
            'morphable_id' => 'nullable|integer',
            'morphable_type' => 'nullable|string',
            'payment_method_id' => 'required|exists:payment_methods,id',
        ]);

        $paymentMethod =PaymentMethod::where('id', $data['payment_method_id'])
            ->sum('total_amount');

        if($paymentMethod < $data['cost']) {
            return $this->apiResponseFailed('Insufficient funds in the selected payment method.');
        }

        $expense = Expense::create($data);
        return $this->apiResponseStored(new ExpenseResource($expense->load(['expenseCategory', 'acceptedBy','paymentMethod'])));
    }
    public function update(Request $request, Expense $expense)
    {
        $data = $request->validate([
            'expense_category_id' => 'sometimes|required|exists:expense_categories,id',
            'bio' => 'sometimes|nullable|string',
            'cost' => 'sometimes|required|numeric',
            'accepted_by' => 'sometimes|nullable|exists:users,id',
            'morphable_id' => 'sometimes|nullable|integer',
            'morphable_type' => 'sometimes|nullable|string',
            'payment_method_id' => 'required|exists:payment_methods,id',
        ]);

        $expense->update($data);
        return $this->apiResponseUpdated(new ExpenseResource($expense->load(['expenseCategory', 'acceptedBy','paymentMethod'])));
    }
    public function destroy(Expense $expense)
    {
        $expense->delete();
        return $this->apiResponseDeleted();
    }

    public function getExpensesStatistics()
    {
        $now = now();
        $startOfMonth = $now->copy()->startOfMonth();
        $endOfMonth = $now->copy()->endOfMonth();

        $startOfLastMonth = $now->copy()->subMonth()->startOfMonth();
        $endOfLastMonth = $now->copy()->subMonth()->endOfMonth();

        $totalExpenses = Expense::sum('cost');
        $monthlyExpenses = Expense::whereBetween('created_at', [$startOfMonth, $endOfMonth])->sum('cost');
        $lastMonthExpenses = Expense::whereBetween('created_at', [$startOfLastMonth, $endOfLastMonth])->sum('cost');
        $totalCount = Expense::count();
        $maxExpense = Expense::max('cost');
        $avgExpense = Expense::avg('cost');

        $groupedExpenses = Expense::with(['expenseCategory'])
            ->select('expense_category_id')
            ->selectRaw('SUM(cost) as total_cost, COUNT(*) as total_expenses')
            ->groupBy('expense_category_id')
            ->get();

        $stats = [
            'total_expenses'         => round($totalExpenses, 2),
            'total_expenses_this_month' => round($monthlyExpenses, 2),
            'total_expenses_last_month' => round($lastMonthExpenses, 2),
            'total_expense_records'  => $totalCount,
            'highest_expense'        => round($maxExpense, 2),
            'average_expense'        => round($avgExpense, 2),
            'categories'=>[]
        ];

        foreach ($groupedExpenses as $category) {
            $stats['categories'][] = [
                'category' => $category->expenseCategory->name,
                'total_cost' => $category->total_cost,
                'total_expenses' => $category->total_expenses,
            ];
        }

        return $this->apiResponseShow([
            'stats' => $stats,
        ]);
    }

}
