<?php

namespace App\Http\Controllers\Api\V1\Instructor;

use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\Evaluation\StudentAttendanceRequest;
use App\Http\Requests\Evaluation\UpdateEvaluationRequest;
use App\Http\Resources\Evaluation\EvaluationResource;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\EnsuresInstructorAccess;
use App\Models\Evaluation;
use App\Models\Lecture;
use App\Models\Round;
use App\Models\RoundStudent;
use App\Models\User;
use Illuminate\Http\JsonResponse;

class InstructorRoundAttendanceController extends Controller
{
    use EnsuresInstructorAccess;

    public function storeStudentAttendance(StudentAttendanceRequest $request, Round $round, Lecture $lecture): JsonResponse
    {
        $this->ensureLectureBelongsToRound($round, $lecture);

        $lecture->load('round.students');
        // Get students in the round
        $studentsInRound = $lecture->round->students->where('type', UserType::STUDENT->value)->pluck('id')->toArray();

        // Filter valid student IDs from the request
        $validStudentIds = array_intersect($request->student_ids, $studentsInRound);

        // Create evaluations
        $evaluations = collect($studentsInRound)->map(function ($student_id) use ($lecture, $validStudentIds) {
            return [
                'lecture_id' => $lecture->id,
                'user_id'    => $student_id,
                'is_attend'  => in_array($student_id, $validStudentIds),
                'created_at' => now(),
                'updated_at' => now(),
            ];
        })->toArray();

        Evaluation::upsert(
            $evaluations,
            ['lecture_id', 'user_id'],
            ['is_attend', 'updated_at']
        );

        // Return the evaluated students
        $evaluatedStudents = User::whereIn('id', $studentsInRound)
            ->with(['evaluation.lecture' => function ($query) {
                $query->with('evaluations');
            }])
            ->get();

        return $this->apiResponseStored(UserResource::collection($evaluatedStudents));
    }

    public function storeStudentEvaluation(UpdateEvaluationRequest $request, Round $round, Lecture $lecture): JsonResponse
    {
        $this->ensureLectureBelongsToRound($round, $lecture);
        $data = $request->validated();
    
        $lecture->load(['round.students']);
    
        // Verify student is in the round
        abort_if(!$lecture->round->students->contains($data['user_id']), 404);
    
        // Get current evaluation
        $currentEvaluation = Evaluation::where('lecture_id', $lecture->id)
            ->where('user_id', $data['user_id'])
            ->first();
    
        // Ensure oldAttendanceDegree is a float
        $oldAttendanceDegree = $currentEvaluation?->attendance_degree ?? 0.0;
    
        // Update evaluation
        $evaluation = Evaluation::updateOrCreate(
            [
                'lecture_id' => $lecture->id,
                'user_id' => $data['user_id']
            ],
            [
                'attendance_degree' => $data['attendance_degree']
            ]
        );
    
        // Update round student points
        $this->updateRoundStudentPoints(
            $lecture->round->id,
            $data['user_id'],
            (float) $data['attendance_degree'],
            (float) $oldAttendanceDegree
        );
    
        return $this->apiResponseStored(EvaluationResource::make($evaluation));
    }

    /**
     * Update the total points for a student in a round based on attendance degree change
     */
    private function updateRoundStudentPoints(int $roundId, int $userId, float $newDegree, float $oldDegree): void
    {
        $roundStudent = RoundStudent::where('round_id', $roundId)
            ->where('user_id', $userId)
            ->first();

        $pointsDifference = $newDegree - $oldDegree;

        if ($pointsDifference > 0) {
            $roundStudent->increment('total_points', $pointsDifference);
        } elseif ($pointsDifference < 0) {
            $roundStudent->decrement('total_points', abs($pointsDifference));
        }
    }
}
