<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Resources\NoticeAbsent\NoticeAbsentResource;
use App\Http\Resources\TaskFeedback\TaskFeedbackResource;
use App\Http\Traits\NotificationTrait;
use App\Models\Lecture;
use App\Enums\FeedbackType;
use App\Models\NoticeAbsent;
use App\Models\TaskFeedback;
use App\Models\TaskSubmission;
use App\Http\Traits\MediaHandler;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Services\ZoomMeetingService;
use App\Http\Resources\Lecture\LectureResource;
use App\Http\Requests\Lecture\CreateLectureRequest;
use App\Http\Requests\Lecture\StoreFeedbackRequest;
use App\Http\Requests\Lecture\UpdateLectureRequest;
use App\Http\Requests\Lecture\GradeSubmissionRequest;
use App\Http\Requests\ZoomIntegration\CreateMeetingRequest;
use App\Http\Resources\TaskSubmission\TaskSubmissionResource;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class LectureController extends Controller
{
    use MediaHandler, NotificationTrait;

    public function index(): AnonymousResourceCollection
    {
        $lectures = Lecture::with([
            'round:id,name',
            'zoomMeetings',
            'attachments',
            'taskSubmissions',
            'lectureTasks',
            'studentEvaluation',
            'studentTaskSubmission',
            'lectureVideos',
        ])
        ->useFilters()
        ->latest()
        ->dynamicPaginate(50);
        return LectureResource::collection($lectures);
    }

    public function store(CreateLectureRequest $request): JsonResponse
    {
        try {
            $data    = $request->validated();
            $lecture = Lecture::create($data);

           if (!empty($data['files'])) {
                foreach ($data['files'] as $file) {
                    $name = $file->getClientOriginalName();
                    $path = $this->upload($file, 'attachments');
                    $lecture->attachments()->create([
                        'name' => $name,
                        'path' => $path,
                    ]);
                }
            }


            if ($this->shouldCreateOrUpdateZoomMeeting($data)) {
            $zoomService = new ZoomMeetingService();
            $zoomService->createMeeting($lecture,$data);
            }
            $this->loadRelations($lecture);
            $this->sendPushNotificationClientSide('admin','add_lecture',$lecture->round_id);
            return $this->apiResponseStored(new LectureResource($lecture));
        } catch (\Exception $e) {
            Log::error('Error creating lecture', ['error' => $e->getMessage()]);
            return $this->apiResponse([],__('main.error',['model' => 'Lecture']),400);
        }
    }

    private function shouldCreateOrUpdateZoomMeeting(array $data): bool
    {
        return isset($data['meeting_start_time']) && isset($data['meeting_duration']);
    }

    public function show(Lecture $lecture): JsonResponse
    {
        $this->loadRelations($lecture);
        return $this->apiResponseShow(new LectureResource($lecture));
    }

    public function update(UpdateLectureRequest $request, Lecture $lecture): JsonResponse
    {
        try {
            $data = $request->validated();
            $lecture->update($data);

            if (!empty($data['files'])) {
                if(!empty($data['remove_files_ids'])){
                    foreach ($lecture->attachments()->whereIn('id',$data['remove_files_ids'])->get() as $attachment) {
                        Storage::disk('public')->delete($attachment->path);
                        $attachment->delete();
                    }
                }

                foreach ($data['files'] as $file) {
                    $name = $file->getClientOriginalName();
                    $path = $this->updateMedia($file, 'attachments');

                    $lecture->attachments()->create([
                        'name' => $name,
                        'path' => $path
                    ]);
                }
            }


            if ($this->shouldCreateOrUpdateZoomMeeting($data)) {
            $zoomService = new ZoomMeetingService();
            if (isset($lecture->zoomMeetings)) {
                $zoomService->updateMeeting($lecture, $lecture->zoomMeetings->ref_id, $data);
            } else {
                    $zoomService->createMeeting($lecture, $data);
                }
            }
            $this->loadRelations($lecture);
            $lecture->refresh();
            return $this->apiResponseUpdated(new LectureResource($lecture));
        } catch (\Exception $e) {
            Log::error('Error updating lecture', ['error' => $e->getMessage()]);
            return $this->apiResponse([],__('main.error',['model' => 'Lecture']),400);
        }
    }

    public function destroy(Lecture $lecture): JsonResponse
    {
        $lecture->delete();
        foreach ($lecture->attachments as $attachment) {
            Storage::disk('public')->delete($attachment->path);
            $attachment->delete();
        }
        return $this->apiResponseDeleted();
    }

    protected function loadRelations(Lecture $lecture): Lecture
    {
        return $lecture->load([
            'round',
            'zoomMeetings',
            'attachments',
            'taskSubmissions',
            'lectureTasks',
            'studentEvaluation',
            'studentTaskSubmission',
        ]);
    }

    public function addMeetingToLecture(Lecture $lecture,CreateMeetingRequest $request): JsonResponse
    {
        try {
            $data         = $request->validated();
            $zoomService  = new ZoomMeetingService();
            if ($lecture->zoomMeetings) {
            $zoomResponse = $zoomService->updateMeeting($lecture, $lecture->zoomMeetings->ref_id, $data);
        } else {
            $zoomResponse = $zoomService->createMeeting($lecture, $data);
        }
        if (isset($zoomResponse['response']['join_url'])) {
            $lecture->update(['url' => $zoomResponse['response']['join_url']]);
        }
        $lecture->load('zoomMeetings');
        return $this->apiResponseUpdated(new LectureResource($lecture));
        } catch (\Exception $e) {
            Log::error('Error adding meeting to lecture', ['error' => $e->getMessage()]);
            return $this->apiResponse([],__('main.error',['model' => 'Lecture']),400);
        }
    }

    public function getLectureRecordings(Lecture $lecture)
    {

    }

    /**
     * Get all submitted tasks for a specific lecture
     *
     * @param Lecture $lecture
     * @return AnonymousResourceCollection
     */
    public function getSubmissionsForReview(Lecture $lecture): AnonymousResourceCollection
    {
        $submissions = TaskSubmission::with(['user','gradedBy','feedbacks'])
        ->where('lecture_id', $lecture->id)
        ->latest()
        ->useFilters()
        ->dynamicPaginate(50);
        return TaskSubmissionResource::collection($submissions);
    }


    public function gradeSubmission(GradeSubmissionRequest $request, Lecture $lecture, TaskSubmission $task)
    {
        // abort_if($task->lecture_id != $lecture->id, 404);
        $data = $request->validated();

        $task = TaskSubmission::where('lecture_id', $lecture->id)
            ->where('user_id', $data['user_id'])
            ->first();

        if ($task) {
            $task->update([
                'task_degree'  => $data['task_degree'],
                'is_corrected' => true,
                'graded_by'    => auth()->id(),
            ]);
        } else {
            $task = TaskSubmission::create([
                'lecture_id'   => $lecture->id,
                'user_id'      => $data['user_id'],
                'task_degree'  => $data['task_degree'],
                'is_corrected' => true,
                'task_url'     => 'https://example.com',
                'graded_by'    => auth()->id(),
            ]);
        }

        $task->load([
            'user',
            'lecture',
            'gradedBy',
            'feedbacks'
        ]);
        return $this->apiResponseUpdated(new TaskSubmissionResource($task));
    }


    public function storeFeedback(StoreFeedbackRequest $request, Lecture $lecture, TaskSubmission $task)
    {
        // abort_if($task->lecture_id != $lecture->id, 404);
        $data = $request->validated();
        if($data['type'] == FeedbackType::IMAGE->value || $data['type'] == FeedbackType::AUDIO->value){
            $data['content'] = $this->upload($data['content'],'tasks/feedback');
        }
        TaskFeedback::create([
            'task_submission_id' => $task->id,
            'content'            => $data['content'],
            'type'               => $data['type'],
            'note'               => $data['note']??null,
        ]);
        $task->load(['user','lecture','gradedBy','feedbacks']);
        return $this->apiResponseUpdated(new TaskSubmissionResource($task));
    }

    public function updateFeedback(StoreFeedbackRequest $request, TaskFeedback $taskFeedback)
    {
        // abort_if($task->lecture_id != $lecture->id, 404);
        $data = $request->validated();
        if($data['type'] == FeedbackType::IMAGE->value || $data['type'] == FeedbackType::AUDIO->value){
            $data['content'] = $this->upload($data['content'],'tasks/feedback');
        }
        $taskFeedback->update([
            'content'            => $data['content'],
            'type'               => $data['type'],
            'note'               => $data['note']??null,
        ]);

        return $this->apiResponseUpdated(new TaskFeedbackResource($taskFeedback));
    }

    public function destroyTaskFeedback(TaskFeedback $feedback): JsonResponse
    {
        $feedback->delete();
        if($feedback->type == FeedbackType::IMAGE->value || $feedback->type == FeedbackType::AUDIO->value){
            MediaHandler::deleteMedia($feedback->content);
        }
        return $this->apiResponseDeleted();
    }

    private function handleTextFeedback(Request $request, TaskSubmission $task, array &$data): void
    {
        $data['feedback'] = $request->input('feedback');
        $data['feedback_type'] = FeedbackType::TEXT->value;
        $this->deleteOldAttachments($task);
    }

    private function handleFileFeedback(Request $request, TaskSubmission $task, array &$data): void
    {
        $this->deleteOldAttachments($task);
        $this->storeFeedbackFiles($request, $task);
        $data['feedback'] = null;
        $data['feedback_type'] = FeedbackType::FILE->value;
    }


    // private function storeFeedbackFiles(Request $request, TaskSubmission $task): void
    // {
    //     $attachments = [];
    //     $note = $request->input('note');

    //     foreach ($request->file('feedback_files') as $file) {
    //         $path = MediaHandler::upload($file, 'tasks/feedback');

    //         // Detect file type (audio or image)
    //         $type = str_starts_with($file->getMimeType(), 'audio/')
    //             ? AttachmentType::AUDIO->value
    //             : AttachmentType::IMAGE->value;

    //         $attachments[] = [
    //             'file_path'           => $path,
    //             'type'                => $type,
    //             'note'                => $note,
    //             'task_submission_id'  => $task->id,
    //             'updated_at'          => now(),
    //         ];
    //     }

    //     // Bulk insert (1 query for all files)
    //     DB::table('task_submission_attachments')->insert($attachments);
    // }


    private function deleteOldAttachments(TaskSubmission $task): void
    {
        foreach ($task->attachments as $attachment) {
            MediaHandler::deleteMedia($attachment->file_path_path);
        }
        $task->attachments()->delete();
    }

    public function getAllNoticeAbsent()
    {
        $noticeAbsents = NoticeAbsent::with(['lecture.round:id,name','student','createdBy','updatedBy'])
            ->latest()
            ->useFilters()
            ->dynamicPaginate(50);
        return NoticeAbsentResource::collection($noticeAbsents);
    }

    public function storeNoticeAbsent(Request $request, Lecture $lecture)
    {
        $data = $request->validate([
            'student_id'=>'required|exists:users,id',
            'content'=>'nullable|string',
            'content_file' => ['nullable', 'required_without:content','file','mimes:webm,wav,mp3,m4a,jpeg,png,jpg', 'max:2048' ],
            'type'         => ['required', 'string', Rule::in(FeedbackType::values())],

        ]);

        if (isset($data['content']) && is_string($data['content'])) {
            $data['content'] = nl2br(trim($data['content']));
        }

        if($data['type'] == FeedbackType::IMAGE->value || $data['type'] == FeedbackType::AUDIO->value){
            $data['content'] = $this->upload($data['content_file'],'notice/absents/');
        }
        $noticeAbsent = NoticeAbsent::create([
            'lecture_id' => $lecture->id,
            'student_id' => $data['student_id'],
            'notice_absent' => $data['content'],
            'type'=> $data['type'],
            'read' => false,
            'created_by' => auth()->id(),
            'updated_by' => auth()->id(),
        ]);
        return $this->apiResponseStored(new NoticeAbsentResource($noticeAbsent->load('lecture:id,name,start_time,created_at','student:id,name','createdBy:id,name','updatedBy:id,name')));
    }

    public function updateNoticeAbsent(Request $request, NoticeAbsent $noticeAbsent)
    {
        $data = $request->validate([
            'content'=>'nullable|string',
            'content_file' => ['nullable', 'required_without:content','file','mimes:webm,wav,mp3,m4a,jpeg,png,jpg', 'max:2048' ],
            'type'         => ['required', 'string', Rule::in(FeedbackType::values())],
        ]);

        if (isset($data['content']) && is_string($data['content'])) {
            $data['content'] = nl2br(trim($data['content']));
        }

        if($data['type'] == FeedbackType::IMAGE->value || $data['type'] == FeedbackType::AUDIO->value){
            $data['content'] = $this->upload($data['content_file'],'notice/absents/');
        }
        $noticeAbsent->update([
            'notice_absent' => $data['content'],
            'type'=>$data['type'],
            'updated_by' => auth()->id(),
        ]);
        return $this->apiResponseUpdated(new NoticeAbsentResource($noticeAbsent->load('lecture:id,name,start_time,created_at','student:id,name','createdBy:id,name','updatedBy:id,name')));
    }

    public function destroyNoticeAbsent(NoticeAbsent $noticeAbsent)
    {
        if($noticeAbsent->type == FeedbackType::IMAGE->value || $noticeAbsent->type == FeedbackType::AUDIO->value){
            MediaHandler::deleteMedia($noticeAbsent->notice_absent);
        }
        $noticeAbsent->delete();
        return $this->apiResponseDeleted();
    }


}
