<?php

namespace App\Http\Controllers\Api\V1;

use App\Http\Controllers\Controller;
use App\Http\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class OrderController extends Controller
{
    use ApiResponse;

    private const ORDER_COLUMN = 'order';
    private const ORDER_DIRECTION_UP = 'up';
    private const ORDER_DIRECTION_DOWN = 'down';

    public function moveUp($id, $model)
    {
        return $this->moveItem($id, $model, self::ORDER_DIRECTION_UP);
    }

    public function moveDown($id, $model)
    {
        return $this->moveItem($id, $model, self::ORDER_DIRECTION_DOWN);
    }

    protected function moveItem($id, $model, $direction)
    {
        try {
            $modelClass = $this->validateModel($model);
            $item = $modelClass::findOrFail($id);

            $this->validateOrderColumn($modelClass);

            $adjacentItem = $this->getAdjacentItem($item, $direction);

            if (!$adjacentItem) {
                $position = $direction === self::ORDER_DIRECTION_UP ? 'top' : 'bottom';
                return $this->apiResponseFailed(__("Item is already at the {$position}"));
            }

            $this->swapOrderValues($item, $adjacentItem);

            return $this->apiResponseUpdated([
                'item' => $item,
                'adjacent_item' => $adjacentItem
            ]);
        } catch (\Exception $e) {
            return $this->apiResponseFailed(__('Failed to update order').': '.$e->getMessage());
        }
    }

    protected function getAdjacentItem($item, $direction)
    {
        return $item->newQuery()
            ->where(self::ORDER_COLUMN, $direction === self::ORDER_DIRECTION_UP ? '<' : '>', $item->{self::ORDER_COLUMN})
            ->orderBy(self::ORDER_COLUMN, $direction === self::ORDER_DIRECTION_UP ? 'desc' : 'asc')
            ->first();
    }

    protected function swapOrderValues($item1, $item2)
    {
        DB::transaction(function () use ($item1, $item2) {
            $tempOrder = $item1->{self::ORDER_COLUMN};
            $item1->{self::ORDER_COLUMN} = $item2->{self::ORDER_COLUMN};
            $item2->{self::ORDER_COLUMN} = $tempOrder;

            $item1->save();
            $item2->save();
        });
    }

    protected function validateModel($model)
    {
        $modelClass = 'App\\Models\\'.Str::studly($model);

        if (!class_exists($modelClass)) {
            throw new \InvalidArgumentException("Model {$model} not found");
        }

        return $modelClass;
    }

    protected function validateOrderColumn($modelClass)
    {
        $instance = new $modelClass;

        if (!in_array(self::ORDER_COLUMN, $instance->getFillable())) {
            throw new \RuntimeException("Model does not have fillable '".self::ORDER_COLUMN."' column");
        }

        if (!\Schema::hasColumn($instance->getTable(), self::ORDER_COLUMN)) {
            throw new \RuntimeException("Table does not have '".self::ORDER_COLUMN."' column");
        }
    }
}
