<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\DefineStatus;
use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentStatus;
use App\Filters\RoundStudentFilters;
use App\Http\Controllers\Controller;
use App\Http\Requests\RoundStudent\AddStudentRequest;
use App\Http\Requests\RoundStudent\ChangeStudentStatusRequest;
use App\Http\Requests\RoundStudent\MoveStudentRequest;
use App\Http\Resources\User\UserResource;
use App\Http\Resources\RoundStudent\RoundStudentResource;
use App\Http\Traits\NotificationTrait;
use App\Models\CourseEnrollment;
use App\Models\Evaluation;
use App\Models\Round;
use App\Models\RoundStudent;
use App\Models\TaskSubmission;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RoundStudentController extends Controller
{
    use NotificationTrait;
    public function addStudent(AddStudentRequest $request)
    {
        $data     = $request->validated();
        $newRound = Round::with('course')->findOrFail($data['round_id']);
        $students = User::whereIn('id', $data['user_ids'])->get();
        foreach ($students as $student) {
            $roundStudent = RoundStudent::where('round_id', $newRound->id)->where('user_id', $student->id)->first();
            if ($roundStudent) {
                continue;
            }

                $amount = $newRound->price ?? $newRound->course->price;
                $newEnrollment = $this->createCourseEnrollmentIfNotExists($newRound, $student, $amount);
                $paymentMethod = $newEnrollment ? 'cash' : 'online';
                $roundStudent =  RoundStudent::create([
                    'round_id'        => $newRound->id,
                    'user_id'         => $student->id,
                    'total_amount'    => $amount,
                    'enrollment_date' => now(),
                    'status'          => DefineStatus::ACTIVE->value,
                    'payment_method'  => $paymentMethod,
                    'payment_status'  => '1',
                    'transaction_id'  => null
                ]);
            }
        $this->sendPushNotificationClientSide('admin','enroll_student',$data['round_id']);
        return $this->apiResponseStored(RoundStudentResource::make($roundStudent));
    }

    public function moveStudent(MoveStudentRequest $request)
    {
        $data = $request->validated();

        $rounds = Round::select('id', 'name')
            ->whereIntegerInRaw('id', [$data['old_round_id'], $data['new_round_id']])
            ->get()
            ->keyBy('id');

        $oldRound = $rounds[$data['old_round_id']] ?? null;
        $newRound = $rounds[$data['new_round_id']] ?? null;


        $enrolledStudents = $oldRound->students()
            ->select('users.id', 'users.name', 'users.email')
            ->whereIntegerInRaw('users.id', $data['user_ids'])
            ->get();

        if ($enrolledStudents->isEmpty()) {
            return $this->apiResponse([__('validation.custom.student_not_enrolled')], 422);
        }

        $studentIds = $enrolledStudents->pluck('id');

        DB::transaction(function () use ($oldRound, $newRound, $studentIds) {
            // Move students to the new round
            $oldRound->students()->detach($studentIds);
            $newRound->students()->attach($studentIds);

            // Update round student counts
            $count = $studentIds->count();
            $oldRound->decrement('students_count', $count);
            $newRound->increment('students_count', $count);
        });

        // Attach the new round to each student for response
        $enrolledStudents->each->setRelation('newRound', $newRound);
        return $this->apiResponseUpdated(UserResource::collection($enrolledStudents));
    }

    public function showStudentsInRound(Round $round)
    {
        $studentsIds = RoundStudent::where('round_id',$round->id)->pluck('user_id');
        $students    = User::whereIn('id',$studentsIds)
            ->useFilters()
            ->with(['studentRounds','evaluations','taskSubmissions','notices.updatedBy','notices.createdBy'])
            ->latest()
            ->dynamicPaginate(50);
        return UserResource::collection($students);
    }

    public function changeStudentStatus(ChangeStudentStatusRequest $request, Round $round, User $user)
    {
        $validated = $request->validated();
        $round->load('students');
        abort_if(!$round->students->contains($user), 404);

        $round->students()->updateExistingPivot($user->id, [
            'status' => $validated['status'],
        ]);

        // Clear the student's progress cache after updating the status
        Cache::forget("student_progress_{$user->id}");

        $user->load(['studentRounds' => function($query) use ($round) {
            $query->where('round_id', $round->id);
        }]);

        return $this->apiResponseStored(new UserResource($user));
    }

    public function rmStudentFromRound(Round $round, User $user)
    {
        $deleted = RoundStudent::where('round_id', $round->id)
            ->where('user_id', $user->id)
            ->delete();

        if (!$deleted) {
            return $this->apiResponse([], __('main.student_not_found_in_round'), 404);
        }

        $courseEnrollments = $round->courseEnrollments()
            ->where('user_id', $user->id)
            ->get();

        foreach ($courseEnrollments as $courseEnrollment) {
            $courseEnrollment->update(['round_id' => null]);
            $courseEnrollment->save();
        }

        $lectureIds = $round->lectures()->pluck('id');
        if ($lectureIds->isNotEmpty()) {
            Evaluation::whereIn('lecture_id', $lectureIds)
                ->where('user_id', $user->id)
                ->delete();
        }

        $taskSubmissions = TaskSubmission::whereHas('lecture', function($query) use ($round) {
                $query->where('round_id', $round->id);
            })
            ->where('user_id', $user->id)
            ->get();

        foreach ($taskSubmissions as $submission) {
            $submission->taskFeedbacks()->delete();
            $submission->delete();
        }

        $user->notifications()
            ->where('data->round_id', $round->id)
            ->delete();

        return $this->apiResponseDeleted();
    }


    private function createCourseEnrollmentIfNotExists(Round $round, User $user, float $amount): ?CourseEnrollment
    {
        $existingEnrollment = CourseEnrollment::where('course_id', $round->course_id)
            ->where('round_id', $round->id)
            ->where('user_id', $user->id)
            ->first();

        if ($existingEnrollment) {
            return null;
        }

        return CourseEnrollment::create([
            'course_id'       => $round->course_id,
            'user_id'         => $user->id,
            'total_amount'    => $amount,
            'enrollment_date' => now(),
            'status'          => true,
            'payment_method'  => PaymentMethodEnum::CASH->value,
            'payment_status'  => PaymentStatus::PENDING->value,
            'transaction_id'  => null,
            'coupon_id'       => null,
            'created_by'      => Auth::id() ?? 1,
            'updated_by'      => Auth::id() ?? 1,
            'round_id'        => $round->id,
            'payment_type'    => 0,
            'coupon_amount'   => 0,
        ]);
    }


    public function unenrolledStudents(Request $request)
    {
        $request->validate([
            'student_ids' => 'required|array',
            'round_id'=> 'required|exists:rounds,id',
        ]);

        $studentIds = $request->input('student_ids');

        $round = Round::findOrFail($request->input('round_id'));

        $deleted = RoundStudent::where('round_id', $round->id)
            ->whereIn('user_id', $studentIds)
            ->delete();

        if ($deleted) {
            foreach ($studentIds as $studentId) {
                $courseEnrollments = $round->courseEnrollments()
                    ->where('user_id', $studentId)
                    ->get();

                foreach ($courseEnrollments as $courseEnrollment) {
                    // $courseEnrollment->installments()->delete();
                    $courseEnrollment->update(['round_id' => null]);
                    $courseEnrollment->round_id = null;
                    $courseEnrollment->save();
                }
            }

            return $this->apiResponseDeleted();
        }

        return $this->apiResponse([], __('main.student_not_found_in_round'), 404);
    }


}
