<?php

namespace App\Http\Controllers\Api\V1\Student;

use App\Enums\DefineStatus;
use App\Http\Resources\Course\CourseResource;
use App\Models\User;
use App\Models\Evaluation;
use App\Enums\FeedbackType;
use App\Models\TaskFeedback;
use App\Models\TaskSubmission;
use Illuminate\Http\JsonResponse;
use App\Http\Controllers\Controller;
use App\Http\Resources\StudentHome\ProgressbarResource;
use App\Http\Resources\TaskFeedback\TaskFeedbackResource;
use App\Http\Resources\TaskSubmission\TaskSubmissionResource;
use App\Models\Round;
use Illuminate\Support\Facades\Cache;

class StudentHomeController extends Controller
{

    public function latestFeedback()
    {
        $userTasks = TaskSubmission::where('user_id', auth()->id())->get();
        $taskIds   = $userTasks->pluck('id');

        // Get the latest feedback for each task
        $feedbacks = TaskFeedback::whereIn('task_submission_id', $taskIds)
            ->latest()
            ->where('type', '!=', FeedbackType::IMAGE)
            ->with(['lecture.round',
             'taskSubmission.user'])
            ->select('task_feedbacks.*')
            ->get()
            ->unique('task_submission_id')
            ->take(10);
        return $this->apiResponseShow(TaskFeedbackResource::collection($feedbacks));
    }


    public function getTaskAllFeedback(TaskSubmission $taskSubmission)
    {
        $feedbacks = TaskFeedback::where('task_submission_id', $taskSubmission->id)
        ->latest()
        ->with('lecture.round',
             'taskSubmission.user')
        ->get();
        return $this->apiResponseShow(TaskFeedbackResource::collection($feedbacks));
    }

    public function getAllFeedback()
    {
        $userTasks = TaskSubmission::where('user_id', auth()->id())->get();
        $taskIds   = $userTasks->pluck('id');

        $taskSubmissions= TaskSubmission::where('user_id', auth()->id())->with(['lecture:id,name,round_id',
        'lecture.round:id,name',
        'feedbacks.createdBy'])->dynamicPaginate(50);

        // $feedbacks = TaskFeedback::whereIn('task_submission_id', $taskIds)
        //     ->latest()
        //     ->with(['lecture',
        //      'taskSubmission.user'])
        //     ->select('task_feedbacks.*')
        //     ->dynamicPaginate()
        //     ->unique('task_submission_id');
        return TaskSubmissionResource::collection($taskSubmissions);
    }

    /**
     * Retrieve a task submission with its attachments.
     *
     * Ensures the authenticated user owns the submission before returning
     * the task details along with all related attachments.
     */
    public function getFeedbackAttachments(TaskSubmission $taskSubmission)
    {
          // Ensure the authenticated user owns this submission
        abort_if($taskSubmission->user_id !== auth()->id(), 403);

        // Load the attachments
        $taskSubmission->load(['attachments' => function ($query) {
            $query->latest();
        }]);

        return new TaskSubmissionResource($taskSubmission);
    }

    /**
     * Get the student's progress in their enrolled rounds.
     * Progress is calculated based on task degrees and attendance degrees.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    // public function studentProgress(): JsonResponse
    // {
    //     $user = auth()->user();

    //     $enrolledRounds = $user->enrolledRounds()
    //         ->with([
    //             'course:id,title',
    //             'lectures:id,round_id' // Load lecture IDs for counting
    //         ])
    //         // ->select(['rounds.id', 'rounds.name', 'rounds.course_id'])
    //         ->withCount('lectures')
    //         ->get();

    //     if ($enrolledRounds->isEmpty()) {
    //         return $this->apiResponseShow([]);
    //     }

    //     $enrolledRounds->each(function ($round) use ($user) {
    //         if ($round) {
    //             // Use lectures_count from withCount
    //             $totalLectures = $round->lectures_count;

    //             // Calculate maximum possible degrees
    //             $maxTaskDegrees = $totalLectures * 10;
    //             $maxAttendanceDegrees = $totalLectures * 10;
    //             $totalPossibleDegrees = $maxTaskDegrees + $maxAttendanceDegrees;

    //             // Get task degrees sum
    //             $earnedTaskDegrees = TaskSubmission::where('user_id', $user->id)
    //                 ->whereIntegerInRaw('lecture_id', $round->lectures->pluck('id'))
    //                 ->sum('task_degree');

    //             // Get attendance degrees sum using a single query
    //             $earnedAttendanceDegrees = Evaluation::where('user_id', $user->id)
    //                 ->whereIntegerInRaw('lecture_id', $round->lectures->pluck('id'))
    //                 ->sum('attendance_degree');

    //             // Calculate progress percentage
    //             $totalEarnedDegrees = $earnedTaskDegrees + $earnedAttendanceDegrees;
    //             $round->total_progress = $totalPossibleDegrees > 0
    //                 ? round(($totalEarnedDegrees / $totalPossibleDegrees) * 100, 2)
    //                 : 0;
    //         }
    //     });

    //     return $this->apiResponseShow(ProgressbarResource::collection($enrolledRounds));
    // }

    // public function studentProgress(): JsonResponse
    // {
    //     $user = auth()->user();
    //     $cacheKey = "student_progress_{$user->id}";

    //     // Use cache to store progress for 1 day
    //     $enrolledRounds = Cache::remember($cacheKey, now()->addDay(1), function () use ($user) {
    //         $enrolledRounds = $user->enrolledRounds()
    //             ->wherePivot('status', DefineStatus::ACTIVE)
    //             ->with([
    //                 'course:id,title',
    //                 'lectures:id,round_id'
    //             ])
    //             ->withCount('lectures')
    //             ->get();

    //         if ($enrolledRounds->isEmpty()) {
    //             return [];
    //         }

    //         $enrolledRounds->each(function ($round) use ($user) {
    //             if ($round) {
    //                 $totalLectures = $round->lectures_count;
    //                 $maxTaskDegrees = $totalLectures * 10;
    //                 $maxAttendanceDegrees = $totalLectures * 10;
    //                 $totalPossibleDegrees = $maxTaskDegrees + $maxAttendanceDegrees;

    //                 $earnedTaskDegrees = TaskSubmission::where('user_id', $user->id)
    //                     ->whereIntegerInRaw('lecture_id', $round->lectures->pluck('id'))
    //                     ->sum('task_degree');

    //                 $earnedAttendanceDegrees = Evaluation::where('user_id', $user->id)
    //                     ->whereIntegerInRaw('lecture_id', $round->lectures->pluck('id'))
    //                     ->sum('attendance_degree');

    //                 $totalEarnedDegrees = $earnedTaskDegrees + $earnedAttendanceDegrees;
    //                 $round->total_progress = $totalPossibleDegrees > 0
    //                     ? round(($totalEarnedDegrees / $totalPossibleDegrees) * 100, 2)
    //                     : 0;
    //             }
    //         });

    //         return $enrolledRounds;
    //     });

    //     return $this->apiResponseShow(ProgressbarResource::collection($enrolledRounds));
    // }

    public function studentProgress(): JsonResponse
    {
        $user = auth()->user();
        $stats = [];
        $rounds=$user->enrolledRounds()->wherePivot('status', DefineStatus::ACTIVE)->get();
        foreach($rounds as $round){
            $stats[] = [
                'total_progress' => round(($this->getAttendanceRate($round) + $round->getTaskSubmissionRateByUser()) / 2,2),
                'round' => [
                    'id' => $round->id,
                    'name' => $round->name,
                ],
                // 'course' => new CourseResource($round->course),
            ];
        }

        return $this->apiResponseShow($stats);
    }


    /**
     * Get student rankings within their enrolled rounds.
     * Ranking is based on attendance and task scores.
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function studentRanking(): JsonResponse
    {
        $user = auth()->user();
        [$rounds, $allStudents] = $this->getRoundsAndStudents($user);

        $ranking = $this->calculateRanking($rounds, $allStudents);

        return $this->apiResponseShow($ranking);
    }

    /**
     * Fetch rounds the student is enrolled in along with all students in those rounds.
     *
     * @param \App\Models\User $user
     * @return array
     */
    private function getRoundsAndStudents($user)
    {
        // Eager load rounds with their lectures
        $rounds = $user->enrolledRounds()->with('lectures')->get();

        // Collect all lecture IDs from all rounds
        $lectureIds = $rounds->flatMap->lectures->pluck('id')->unique();

        // Get all students in any of these rounds with their relevant data
        $allStudents = User::whereHas('enrolledRounds', function ($query) use ($rounds) {
            $query->whereIntegerInRaw('round_id', $rounds->pluck('id'));
        })->with([
            'taskSubmissions' => function ($query) use ($lectureIds) {
                $query->whereIntegerInRaw('lecture_id', $lectureIds);
            },
            'evaluations' => function ($query) use ($lectureIds) {
                $query->whereIntegerInRaw('lecture_id', $lectureIds);
            },
            'enrolledRounds'
        ])->get();

        return [$rounds, $allStudents];
    }

    /**
     * Calculate student rankings for each round.
     * Scores are computed based on attendance and submitted task degrees.
     *
     * @param \Illuminate\Support\Collection $rounds
     * @param \Illuminate\Support\Collection $allStudents
     * @return \Illuminate\Support\Collection
     */
    private function calculateRanking($rounds, $allStudents)
    {
        return $rounds->map(function ($round) use ($allStudents) {
            // Get students in this specific round
            $students = $allStudents->filter(fn($student) =>
                $student->enrolledRounds->contains('id', $round->id)
            );

            // Get lecture IDs for current round
            $roundLectureIds = $round->lectures()->whereHas('taskSubmissions')->pluck('id');
            $roundLectureIdsCount=$roundLectureIds->count();

            // Calculate scores
            $studentScores = $students->map(function ($student) use ($roundLectureIdsCount,$roundLectureIds) {

                $attendancePoints = $student->evaluations
                    ->whereIn('lecture_id', $roundLectureIds)
                    ->where('is_attend', true)
                    ->count();

                $attendancePointsPer = $roundLectureIdsCount
                        ? round(($attendancePoints / $roundLectureIdsCount) * 100, 2)
                        : 0;

                $taskDegree = $student->taskSubmissions
                    ->whereIn('lecture_id', $roundLectureIds)
                    ->sum('task_degree');

                $taskDegreePer = $roundLectureIdsCount
                    ? round(($taskDegree / ($roundLectureIdsCount * 10)) * 100, 2)
                    : 0;

                return [
                    'student_id' => $student->id,
                    'name' => $student->name,
                    'attendance_points'=>$attendancePointsPer,
                    'taskDegree' => $taskDegreePer,
                    'score' => min(100, round((($attendancePointsPer + $taskDegreePer) / 2) + $student->bonus, 2)),
                ];
            });

            return [
                'round_id' => $round->id,
                'round_name' => $round->name,
                'top_students' => $studentScores->sortByDesc('score')->values(),
            ];
        });
    }


    public function stats(){
        $user = auth()->user();
        $stats = [];
        $rounds=$user->enrolledRounds()->wherePivot('status', DefineStatus::ACTIVE)->get();
        foreach($rounds as $round){
            $stats[] = [
                'round' => $round->name,
                'attendance_rate' => $this->getAttendanceRate($round),
                'task_submission_rate' => $round->getTaskSubmissionRateByUser(),
                'score' => min(100, round((($this->getAttendanceRate($round) + $round->getTaskSubmissionRateByUser()) / 2) + $user->bonus, 2)),
                'average_rating' => $this->getAverageRating($round),
                'bonus' => $user->bonus,
            ];
        }

        return $this->apiResponseShow($stats);
    }


    public function getAverageRating(Round $round){
        $lectureIds = $round->lectures()->pluck('lectures.id');
        $taskSubmissions = TaskSubmission::where('user_id',auth()->id())->whereIn('lecture_id', $lectureIds)->get();
        $totalEvaluations = $taskSubmissions->sum('task_degree');
        $lectureCount=($lectureIds->count()*10);

        return round(($totalEvaluations / $lectureCount) * 100, 2);
    }


    public function getAttendanceRate(Round $round)
    {
        $lectureIds = $round->lectures()->pluck('lectures.id');

        $totalLectures = $lectureIds->count();

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->where('user_id', auth()->id())
            ->where('is_attend', true)
            ->count();

        return round(($actualAttendances / $totalLectures) * 100, 2);
    }

    public function statsCharts()
    {
        $user = auth()->user();
        $stats = [];

        $rounds = $user->enrolledRounds()->wherePivot('status', DefineStatus::ACTIVE)->get();

        foreach ($rounds as $round) {
            $round->lectures->each(function ($lecture) use (&$stats) {
                $taskSubmission = $lecture->taskSubmissions()->where('user_id', auth()->id())->first();
                if ($taskSubmission) {
                    $stats[$lecture->name] = ["degrees"=>$taskSubmission->task_degree,"lecture_date"=>$lecture->start_time];
                } else {
                    $stats[$lecture->name] = ["degrees"=>null,"lecture_date"=>$lecture->start_time];
                }
            });
        }

        return $this->apiResponse($stats);
    }


}
