<?php

namespace App\Http\Controllers\Api\V1\Student;

use App\Http\Controllers\Controller;
use App\Http\Requests\Student\Ticket\AddMessageRequest;
use App\Http\Requests\Student\Ticket\StoreTicketRequest;
use App\Http\Resources\Ticket\TicketResource;
use App\Http\Resources\TicketMessage\TicketMessageResource;
use App\Http\Traits\MediaHandler;
use App\Models\Ticket;
use App\Services\TicketService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class StudentTicketController extends Controller
{
    use MediaHandler;

    protected TicketService $ticketService;

    public function __construct(TicketService $ticketService)
    {
        $this->ticketService = $ticketService;
    }

    /**
     * Get all tickets and filter it based on status.
     */
    public function getAllTickets(): AnonymousResourceCollection
    {
        $user = auth()->user();

        $tickets = Ticket::select('id', 'description', 'status', 'created_at')
            ->where('user_id', $user->id)
            ->useFilters()
            ->dynamicPaginate();

        return TicketResource::collection($tickets);
    }

    /**
     * Get single ticket.
     */
    public function showSingleTicket(Ticket $ticket): JsonResponse
    {
        $user = auth()->user();

        if ($ticket->user_id !== $user->id) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }

        $ticket->load('messages');

        return $this->apiResponseShow(new TicketResource($ticket));
    }

    /**
     * Store a new ticket and return the stored response.
     *
     * @param StoreTicketRequest $request The validated request data.
     * @return JsonResponse The API response with the stored ticket data.
     */
    public function storeTicket(StoreTicketRequest $request): JsonResponse
    {
        $ticket = $this->ticketService->createTicket($request->validated());

        return $this->apiResponseStored(new TicketResource($ticket));
    }

    /**
     * Add a content message to an existing ticket.
     *
     * @param AddMessageRequest $request The validated request data.
     * @param Ticket $ticket The ticket to which the message will be added.
     * @return JsonResponse The API response with the stored message data.
     */
    public function addContentMessage(AddMessageRequest $request, Ticket $ticket): JsonResponse
    {
        $message = $this->ticketService->addMessage($ticket, $request->validated());

        return $this->apiResponseStored(new TicketMessageResource($message));
    }
}
