<?php

namespace App\Http\Controllers\Api\V1\Student;
use App\Enums\TicketPriority;
use App\Enums\TicketStatus;
use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Resources\TicketSystem\TicketSystemResource;
use App\Http\Traits\MediaHandler;
use App\Models\Ticket;
use App\Models\TicketAttachment;
use App\Models\TicketReply;
use App\Models\TicketSystem;
use Illuminate\Http\Request;
use Kreait\Firebase\Factory;
class TicketSystemController extends Controller
{
    use MediaHandler;
    public $database;

    public function __construct()
    {
        if (env('FIREBASE_CREDENTIALS')) {

        $factory = (new Factory)
                ->withServiceAccount(config('services.firebase.credentials'))
                ->withDatabaseUri(config('services.firebase.database_url'));
            $this->database = $factory->createDatabase();
        }
        else{
            // $this->database = null;
        }
    }
    public function index(){
        $ticketSystems=TicketSystem::where('user_id',auth()->id())->useFilters()->dynamicPaginate();
        return TicketSystemResource::collection($ticketSystems->load('assignedTo'));
    }
    public function store(Request $request){
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string|max:10000000',
            'priority' => 'nullable|integer|in:' . implode(',', TicketPriority::values()),
            'status'   => 'nullable|integer|in:' . implode(',', TicketStatus::values()),
            'ticket_attachments' => 'nullable|array',
            'ticket_attachments.*.id' => 'nullable|integer',
            'ticket_attachments.*.file_path' => 'nullable|file'
        ]);

        $ticketSystem=TicketSystem::create([
            'subject' => $request->subject,
            'description' => $request->description,
            'priority' => $request->priority ?? TicketPriority::LOW->value,
            'status' => $request->status ?? TicketStatus::NEW->value,
        ]);

        $ticket_attachments = $request->ticket_attachments ?? [];
        if (!empty($ticket_attachments)) {
            $this->handleTicketAttachments($ticketSystem, $ticket_attachments);
        }

        try {
            $messageData = [
                'id' => $ticketSystem->id,
                'ticket_id' => $ticketSystem->id,
                'user_id' => auth()->id(),
                'user_name' => auth()->user()->name,
                'status' => TicketStatus::from($ticketSystem->status)->name,
                'type'=>UserType::userType(UserType::from(auth()->user()->type)),
                'message' => $request->description,
                'created_at' => now()->toDateTimeString(),
            ];
            $this->database
                ->getReference("tickets/{$ticketSystem->id}/messages")
                ->push($messageData);
        } catch (\Exception $e) {
            \Log::error('Failed to save message to Firebase', [
                'error' => $e->getMessage(),
                'ticket_id' => $ticketSystem->id,
                'user_id' => auth()->id()
            ]);
        }

        return $this->apiResponseStored(new TicketSystemResource($ticketSystem));
    }

    public function show(TicketSystem $ticketSystem){
        return $this->apiResponseShow(new TicketSystemResource($ticketSystem->load('assignedTo','ticketReplies.user','ticketAttachments')));
    }

    public function update(TicketSystem $ticketSystem, Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string|max:10000000',
            'priority' => 'nullable|integer|in:' . implode(',', TicketPriority::values()),
            'status'   => 'nullable|integer|in:' . implode(',', TicketStatus::values()),
            'ticket_replies' => 'nullable|array',
            'ticket_replies.*.id' => 'nullable|integer',
            'ticket_replies.*.message' => 'nullable|string|max:10000000',
            'ticket_attachments' => 'nullable|array',
            'ticket_attachments.*.id' => 'nullable|integer',
            'ticket_attachments.*.file_path' => 'nullable|file'
        ]);

        $ticketSystem->update([
            'subject' => $request->subject,
            'description' => $request->description,
            'priority' => $request->priority ?? $ticketSystem->priority,
            'status' => $request->status ?? $ticketSystem->status,
        ]);

        $ticket_replies = $request->ticket_replies ?? [];
        if (!empty($ticket_replies)) {
            $this->handleTicketReplies($ticketSystem, $ticket_replies);
        }

        $ticket_attachments = $request->ticket_attachments ?? [];
        if (!empty($ticket_attachments)) {
            $this->handleTicketAttachments($ticketSystem, $ticket_attachments);
        }

        return $this->apiResponseUpdated(new TicketSystemResource($ticketSystem->load('assignedTo','ticketReplies.user','ticketAttachments')));
    }

    protected function handleTicketReplies($ticketSystem, $ticket_replies)
    {
        $ticket_reply_ids = collect($ticket_replies)->pluck('id')->filter()->toArray();

        TicketReply::where('ticket_id', $ticketSystem->id)
            ->where('user_id', auth()->id())
            ->whereNotIn('id', $ticket_reply_ids)
            ->delete();

        foreach ($ticket_replies as $reply) {
            if (empty($reply['message'])) {
                continue;
            }

            $payload = [
                'ticket_id' => $ticketSystem->id,
                'user_id' => auth()->id(),
                'message' => $reply['message'],
            ];

            if (!empty($reply['id'])) {
                $ticketReply = TicketReply::where('id', $reply['id'])
                    ->where('user_id', auth()->id())
                    ->first();
                if ($ticketReply) {
                    $ticketReply->update($payload);
                }
            } else {
                TicketReply::create($payload);
            }
        }
    }

    protected function handleTicketAttachments($ticketSystem, $ticket_attachments)
    {
        $ticket_attachment_ids = collect($ticket_attachments)
            ->pluck('id')
            ->filter()
            ->toArray();

        TicketAttachment::where('ticket_id', $ticketSystem->id)
            ->where('user_id', auth()->id())
            ->whereNotIn('id', $ticket_attachment_ids)
            ->delete();

        foreach ($ticket_attachments as $attachment) {
            $payload = [
                'ticket_id' => $ticketSystem->id,
                'user_id' => auth()->id(),
            ];

            if (!empty($attachment['id'])) {
                $ticketAttachment = TicketAttachment::where('id', $attachment['id'])
                    ->where('user_id', auth()->id())
                    ->first();

                if ($ticketAttachment) {
                    if (isset($attachment['file_path'])) {
                        $payload['file_path'] = $this->updateMedia(
                            $attachment['file_path'],
                            'tickets',
                            $ticketAttachment->file_path
                        );
                    }
                    $ticketAttachment->update($payload);
                }
            }
            elseif (isset($attachment['file_path']) && $attachment['file_path']) {
                $payload['file_path'] = $this->upload($attachment['file_path'], 'tickets');
                TicketAttachment::create($payload);
            }
        }
    }

    public function destroy(TicketSystem $ticketSystem){
        $ticketSystem->ticketReplies()->each(function($ticketReply){
            $ticketReply->delete();
        });
        $ticketSystem->ticketAttachments()->each(function($ticketAttachment){
            $this->deleteMedia($ticketAttachment->file_path);
            $ticketAttachment->delete();
        });
        $ticketSystem->delete();
        return $this->apiResponseDeleted();
    }


    public function getReplies($ticket_id)
    {
        $ticket = TicketSystem::findOrFail($ticket_id);
        $replies = $ticket->ticketReplies()->with('user:id,name')->orderBy('created_at', 'desc')->get();

        return $this->apiResponseShow($replies);
    }

    public function storeReply(Request $request, $ticket_id)
    {
        $request->validate([
            'content' => 'required|string|max:10000000',
        ]);

        $reply = TicketReply::create([
            'ticket_id' => $ticket_id,
            'user_id' => auth()->id(),
            'message' => $request->content,
        ]);
        $messageData = [
            'id' => $reply->id,
            'ticket_id' => $ticket_id,
            'user_id' => auth()->id(),
            'user_name' => auth()->user()->name,
            'type'=>UserType::userType(UserType::from(auth()->user()->type)),
            'message' => $request->content,
            'created_at' => now()->toDateTimeString(),
        ];

        try {
            $this->database
                ->getReference("tickets/{$ticket_id}/messages")
                ->push($messageData);
        } catch (\Exception $e) {
            \Log::error('Failed to save message to Firebase', [
                'error' => $e->getMessage(),
                'ticket_id' => $ticket_id,
                'user_id' => auth()->id()
            ]);
        }

        return $this->apiResponseStored($reply->load('user:id,name'));
    }

    public function updateReply(Request $request, $ticket_id, $reply_id)
    {
        $request->validate([
            'content' => 'required|string|max:10000000',
        ]);

        $reply = TicketReply::where('id', $reply_id)
            ->where('ticket_id', $ticket_id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $reply->update([
            'message' => $request->content,
        ]);

        return $this->apiResponseUpdated($reply->load('user:id,name'));
    }

    public function destroyReply($ticket_id, $reply_id)
    {
        $reply = TicketReply::where('id', $reply_id)
            ->where('ticket_id', $ticket_id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $reply->delete();
        return $this->apiResponseDeleted();
    }

}
