<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\DefineStatus;
use App\Enums\PaymentMethodEnum;
use App\Enums\PaymentStatus;
use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Requests\User\CreateUserRequest;
use App\Http\Requests\User\UpdateUserRequest;
use App\Http\Resources\User\UserResource;
use App\Http\Traits\MediaHandler;
use App\Models\CourseEnrollment;
use App\Models\Round;
use App\Models\RoundStudent;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
class StudentController extends Controller
{
    use MediaHandler;

    public function index(): AnonymousResourceCollection
    {
        $students = User::Students()
            ->with(['studentRounds','evaluations','evaluation','enrolledRounds.lectures','enrolledCourses','transactions','installments.user','taskSubmissions','taskFeedbacks','notices.updatedBy','notices.createdBy','absencesNotices','skills'])
            ->useFilters()
            ->latest()
            ->dynamicPaginate(40);
        return UserResource::collection($students);
    }

    public function client(): AnonymousResourceCollection
    {
        $students = User::Students()
            ->whereHas('enrolledCourses')
            ->useFilters()
            ->latest()
            ->dynamicPaginate(40);
        return UserResource::collection($students);
    }

    public function store(CreateUserRequest $request): JsonResponse
    {
        $data = $request->validated();

        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::upload($request->file('photo'), 'student/photos')
            : null;

        $data['type'] = UserType::STUDENT;
        $data['password']=Hash::make($data['password']);
        $student = User::create($data);
        return $this->apiResponseStored(new UserResource($student));
    }

    public function show(User $student): JsonResponse
    {
        if (!$student || $student->type !== UserType::STUDENT->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        return $this->apiResponseShow(new UserResource($student->load('enrolledCourses','studentRounds')));
    }

    public function update(UpdateUserRequest $request, User $student): JsonResponse
    {
        if (!$student || $student->type !== UserType::STUDENT->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }
        $data = $request->validated();
        $data['photo'] = $request->hasFile('photo')
            ? MediaHandler::updateMedia($data['photo'], 'student/photos', $student->photo)
            : $student->photo;

       $data['password'] = collect($data)->get('password')
        ? Hash::make($data['password'])
        : $student->password;

        $student->update($data);
        return $this->apiResponseUpdated(new UserResource($student));
    }

    public function destroy(User $student): JsonResponse
    {
        if (!$student || $student->type !== UserType::STUDENT->value) {
            return $this->apiResponse([], __('main.record_not_found'), 404);
        }

         if ($student->studentRounds()->exists()) {
            return $this->apiResponse([], __('main.cannot_delete_student_with_rounds'), 400);
        }

        if($student->enrolledCourses()->exists()) {
            return $this->apiResponse([], __('main.cannot_delete_student_with_enrollments'), 400);
        }

        $student->photo ? MediaHandler::deleteMedia($student->photo_path) : null;

        $student->delete();

        return $this->apiResponseDeleted();
    }

    public function enrollments()
    {
        // $unenrolledStudents = User::students()
        //     ->useFilters()
        //     ->latest()
        //     ->dynamicPaginate();
        // return UserResource::collection();
    }
    public function changePlacementTestStatus(Request $request)
    {
        $validated = $request->validate([
            'user_id' => 'required|exists:users,id',
            'placement_test_status' => 'required',
        ]);

        $user = User::findOrFail($validated['user_id']);
        $user->placement_test_status = $validated['placement_test_status'];
        $user->save();

        return $this->apiResponseUpdated([], __('main.placement_test_status_changed'));
    }

    public function enrolledStudents(Request $request){
        $data=$request->validate([
            'student_id'=> 'required|exists:users,id',
            'round_id'=> 'required|exists:rounds,id'
        ]);
        $newRound = Round::with('course')->findOrFail($data['round_id']);

        $user = User::findOrFail($data['student_id']);

        $roundStudent = RoundStudent::where('round_id', $newRound->id)
                                    ->where('user_id', $user->id)
                                    ->first();

        if($roundStudent){
            return $this->apiResponseUpdated([], __('main.student_already_enrolled'));
        }

        $amount = $newRound->price ?? $newRound->course->price;
        CourseEnrollment::create([
            'course_id'       => $newRound->course_id,
            'user_id'         => $user->id,
            'total_amount'    => $amount,
            'enrollment_date' => now(),
            'status'          => true,
            'payment_method'  => PaymentMethodEnum::CASH->value ,
            'payment_status'  => PaymentStatus::PENDING->value,
            'transaction_id'  => null,
            'coupon_id'       => null,
            'created_by'      => Auth::id() ?? 1,
            'updated_by'      => Auth::id() ?? 1,
            'round_id'        => $newRound->id,
            'payment_type'    => 0,
            'coupon_amount'   => 0,
        ]);
        RoundStudent::create([
            'round_id'        => $newRound->id,
            'user_id'         => $user->id,
            'total_amount'    => $amount,
            'enrollment_date' => now(),
            'status'          => DefineStatus::ACTIVE->value,
            'payment_method'  => 'cache',
            'payment_status'  => '1',
            'transaction_id'  => null
        ]);

        return $this->apiResponseUpdated([]);
    }

    public function addBonus(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'bonus'   => 'required|integer|min:0|max:100',
        ]);

        $user = User::findOrFail($request->user_id);

        $user->update([
            'bonus' => $request->bonus,
        ]);
        $user->save();

        return $this->apiResponseUpdated([]);
    }


}
