<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\SurveyQuestionType;
use App\Http\Controllers\Controller;
use App\Http\Resources\SurveyAnswerResource;
use App\Http\Resources\SurveyResource;
use App\Http\Traits\MediaHandler;
use App\Models\Survey;
use App\Models\SurveyQuestion;
use Illuminate\Http\Request;

class SurveyController extends Controller
{
    use MediaHandler;
    public function index()
    {
        $surveys = Survey::with(['sections.questions'])->paginate(30);
        return SurveyResource::collection($surveys);
    }

    public function show(Survey $survey)
    {
        $survey->load(['sections.questions']);
        return $this->apiResponseShow(new SurveyResource($survey));
    }

    public function responses(Survey $survey){
        return SurveyAnswerResource::collection($survey->answers()->with(['section:id,title','question:id,question_text,question_type','user:id,name,email'])->paginate(30));
    }

    public function publicSurvey(Survey $survey){
        $survey->load(['sections.questions']);
        return $this->apiResponseShow(new SurveyResource($survey));
    }
    public function publicSurveyAnswer(Request $request, Survey $survey)
    {
        $request->validate([
            'answers' => 'required|array',
            'answers.*.survey_question_id' => 'required|exists:survey_questions,id',
            'answers.*.answer' => 'nullable',
        ]);

        $now = now();
        $answers = [];

        foreach ($request->answers as $answerData) {
            $question = SurveyQuestion::findOrFail($answerData['survey_question_id']);
            $questionType = $question->question_type;

            $answers[] = [
                'survey_id' => $survey->id,
                'survey_section_id' => $question->survey_section_id,
                'survey_question_id' => $answerData['survey_question_id'],
                'answer' => $this->formatAnswer($answerData['answer'] ?? null, $questionType),
                'question_type' => $questionType->value,
                'created_at' => $now,
                'updated_at' => $now,
            ];
        }

        $survey->answers()->createMany($answers);

        return $this->apiResponseStored('Survey answers submitted successfully');
    }

    protected function formatAnswer($answer, SurveyQuestionType $type)
    {
        if (is_null($answer)) {
            return [];
        }

        if (in_array($type, [SurveyQuestionType::FILE, SurveyQuestionType::IMAGE])) {
            $result = $this->handleFileUpload($answer);
            return $result ? [$result] : [];
        }

        if (in_array($type, [
            SurveyQuestionType::CHECKBOX,
            SurveyQuestionType::MATRIX,
            SurveyQuestionType::RANKING,
            SurveyQuestionType::SELECT,
            SurveyQuestionType::DROPDOWN
        ])) {
            return is_array($answer) ? $answer : [$answer];
        }

        return [$answer];
    }
    protected function handleFileUpload($file)
    {
        if (empty($file)) {
            return null;
        }

        if (is_string($file)) {
            return $file;
        }

        try {
            $path = 'survey_attachments/' . date('Y/m/d');
            return $this->upload($file, $path);
        } catch (\Exception $e) {
            return null;
        }
    }
}
