<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\TicketStatus;
use App\Http\Controllers\Controller;
use App\Http\Requests\Ticket\ChangeStatusRequest;
use App\Http\Requests\Ticket\ReplyToTicketRequest;
use App\Http\Resources\Ticket\TicketResource;
use App\Http\Resources\TicketMessage\TicketMessageResource;
use App\Http\Traits\MediaHandler;
use App\Models\Ticket;
use App\Services\TicketService;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class TicketController extends Controller
{
    use MediaHandler;

    protected TicketService $ticketService;

    public function __construct(TicketService $ticketService)
    {
        $this->ticketService = $ticketService;
    }

    /**
     * Get all tickets and filter it based on status.
     */
    public function getAllTickets(): AnonymousResourceCollection
    {
        $tickets = Ticket::select('id', 'description', 'status', 'user_id', 'created_at')
            ->with('student:id,name,email')
            ->useFilters()
            ->dynamicPaginate();

        return TicketResource::collection($tickets);
    }

    /**
     * Get single ticket.
     */
    public function showSingleTicket(Ticket $ticket): JsonResponse
    {
        $ticket->load('messages');
        
        return $this->apiResponseShow(new TicketResource($ticket));
    }

    /**
     * Reply to a ticket as an admin and assign it if pending.
     *
     * @param ReplyToTicketRequest $request The validated request data.
     * @param Ticket $ticket The ticket being replied to.
     * @return JsonResponse The API response with the stored message data or an error response.
     */
    public function replyToTicket(ReplyToTicketRequest $request, Ticket $ticket): JsonResponse
    {
        $admin = auth()->user();

        // Assign the ticket to the admin if it's pending
        $this->ticketService->assignAdmin($ticket, $admin->id);

        // Prevent other admins from replying
        if ($ticket->admin_id && $ticket->admin_id !== $admin->id) return $this->apiResponse(__('validation.custom.ticket_assigned'), 403);

        // Prevent reopening a closed ticket
        if ($ticket->status === TicketStatus::CLOSED) return $this->apiResponse([__('validation.custom.ticket_closed')], 422);

        $message = $this->ticketService->addMessage($ticket, $request->validated());
        return $this->apiResponseStored(new TicketMessageResource($message));
    }

    public function updateStatus(ChangeStatusRequest $request, Ticket $ticket)
    {
        $validated = $request->validated();
        $ticket->update(['status' => $validated['status']]);
        return $this->apiResponseUpdated(new TicketResource($ticket));
    }
}
