<?php

namespace App\Http\Controllers\Api\V1;

use App\Services\WaSenderService;
use Illuminate\Support\Facades\Cache;
use App\Enums\TicketPriority;
use App\Enums\TicketStatus;
use App\Enums\TicketType;
use App\Enums\UserType;
use App\Http\Controllers\Controller;
use App\Http\Resources\TicketSystem\TicketSystemResource;
use App\Http\Traits\MediaHandler;
use App\Models\Ticket;
use App\Models\TicketAttachment;
use App\Models\TicketReply;
use App\Models\TicketSystem;
use App\Models\User;
use Illuminate\Http\Request;
use Kreait\Firebase\Factory;

class TicketSystemController extends Controller
{
    use MediaHandler;
    public $database;

    public function __construct()
    {
        if (env('FIREBASE_CREDENTIALS')) {  
        $factory = (new Factory)
                ->withServiceAccount(config('services.firebase.credentials'))
                ->withDatabaseUri(config('services.firebase.database_url'));

            $this->database = $factory->createDatabase();
        }
        else{
            // $this->database = null;
        }
    }
    public function index(){
        $ticketSystems=TicketSystem::useFilters()->dynamicPaginate();
        return TicketSystemResource::collection($ticketSystems->load('assignedTo','assignedToBy','user'));
    }
    public function getByAssignTo(){
        $ticketSystems=TicketSystem::where('assigned_to',auth()->id())->useFilters()->dynamicPaginate();
        return TicketSystemResource::collection($ticketSystems->load('assignedTo','assignedToBy','user'));
    }
    public function store(Request $request){
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string|max:10000000',
            'priority' => 'nullable|integer|in:' . implode(',', TicketPriority::values()),
            'status'   => 'nullable|integer|in:' . implode(',', TicketStatus::values()),
            'ticket_attachments' => 'nullable|array',
            'ticket_attachments.*.id' => 'nullable|integer',
            'ticket_attachments.*.file_path' => 'nullable|file'
        ]);

        $ticketSystem=TicketSystem::create([
            'subject' => $request->subject,
            'description' => $request->description,
            'priority' => $request->priority ?? TicketPriority::LOW->value,
            'status' => $request->status ?? TicketStatus::PENDING->value,
        ]);

        $ticket_attachments = $request->ticket_attachments ?? [];
        if (!empty($ticket_attachments)) {
            $this->handleTicketAttachments($ticketSystem, $ticket_attachments);
        }

        return $this->apiResponseStored(new TicketSystemResource($ticketSystem));
    }

    public function show(TicketSystem $ticketSystem){
        return $this->apiResponseShow(new TicketSystemResource($ticketSystem->load('assignedTo','assignedToBy','ticketReplies.user','ticketAttachments','user')));
    }

    public function update(TicketSystem $ticketSystem, Request $request)
    {
        $request->validate([
            'subject' => 'required|string|max:255',
            'description' => 'required|string|max:10000000',
            'priority' => 'nullable|integer|in:' . implode(',', TicketPriority::values()),
            'status'   => 'nullable|integer|in:' . implode(',', TicketStatus::values()),
            'assigned_to' => 'nullable|integer|exists:users,id',
            'ticket_replies' => 'nullable|array',
            'ticket_replies.*.id' => 'nullable|integer',
            'ticket_replies.*.message' => 'nullable|string|max:10000000',
            'ticket_attachments' => 'nullable|array',
            'ticket_attachments.*.id' => 'nullable|integer',
            'ticket_attachments.*.file_path' => 'nullable|file'
        ]);

        $ticketSystem->update([
            'subject' => $request->subject,
            'description' => $request->description,
            'priority' => $request->priority ?? $ticketSystem->priority,
            'status' => $request->status ?? $ticketSystem->status,
            'assigned_to'=> $request->assigned_to ?? $ticketSystem->assigned_to,
            'assigned_to_by'=> $request->assigned_to !=null ? auth()->id() : null
        ]);

        $ticket_replies = $request->ticket_replies ?? [];
        if (!empty($ticket_replies)) {
            $this->handleTicketReplies($ticketSystem, $ticket_replies);
        }

        $ticket_attachments = $request->ticket_attachments ?? [];
        if (!empty($ticket_attachments)) {
            $this->handleTicketAttachments($ticketSystem, $ticket_attachments);
        }

        return $this->apiResponseUpdated(new TicketSystemResource($ticketSystem->load('assignedTo','assignedToBy','ticketReplies','ticketAttachments')));
    }

    protected function handleTicketReplies($ticketSystem, $ticket_replies)
    {
        $ticket_reply_ids = collect($ticket_replies)->pluck('id')->filter()->toArray();

        TicketReply::where('ticket_id', $ticketSystem->id)
            ->where('user_id', auth()->id())
            ->whereNotIn('id', $ticket_reply_ids)
            ->delete();

        foreach ($ticket_replies as $reply) {
            if (empty($reply['message'])) {
                continue;
            }

            $payload = [
                'ticket_id' => $ticketSystem->id,
                'user_id' => auth()->id(),
                'message' => $reply['message'],
            ];

            if (!empty($reply['id'])) {
                $ticketReply = TicketReply::where('id', $reply['id'])
                    ->where('user_id', auth()->id())
                    ->first();
                if ($ticketReply) {
                    $ticketReply->update($payload);
                }
            } else {
                TicketReply::create($payload);
            }
        }
    }

    protected function handleTicketAttachments($ticketSystem, $ticket_attachments)
    {
        $ticket_attachment_ids = collect($ticket_attachments)
            ->pluck('id')
            ->filter()
            ->toArray();

        TicketAttachment::where('ticket_id', $ticketSystem->id)
            ->where('user_id', auth()->id())
            ->whereNotIn('id', $ticket_attachment_ids)
            ->delete();

        foreach ($ticket_attachments as $attachment) {
            $payload = [
                'ticket_id' => $ticketSystem->id,
                'user_id' => auth()->id(),
            ];

            if (!empty($attachment['id'])) {
                $ticketAttachment = TicketAttachment::where('id', $attachment['id'])
                    ->where('user_id', auth()->id())
                    ->first();

                if ($ticketAttachment) {
                    if (isset($attachment['file_path'])) {
                        $payload['file_path'] = $this->updateMedia(
                            $attachment['file_path'],
                            'tickets',
                            $ticketAttachment->file_path
                        );
                    }
                    $ticketAttachment->update($payload);
                }
            }
            elseif (isset($attachment['file_path']) && $attachment['file_path']) {
                $payload['file_path'] = $this->upload($attachment['file_path'], 'tickets');
                TicketAttachment::create($payload);
            }
        }
    }

    public function assignTicket(Request $request, TicketSystem $ticketSystem){
        $request->validate([
            'assigned_to' => 'required|integer|exists:users,id',
        ]);
        $ticketSystem->update([
            'status'=>TicketStatus::IN_PROGRESS->value,
            'assigned_to' => $request->assigned_to,
            'assigned_to_by' => auth()->id(),
        ]);

        $user=User::find($request->assigned_to);
        if($user && $user->type==UserType::INSTRUCTOR->value){
            $waSenderService=new WaSenderService();
            $waSenderService->sendMessage($user->phone, 'Ticket assigned to you');
        }

        return $this->apiResponseUpdated($ticketSystem->load('assignedTo:id,name','assignedToBy:id,name'));
    }

    public function changeTicketStatus(Request $request, TicketSystem $ticketSystem){
        $request->validate([
            'status' => 'required|integer|in:' . implode(',', TicketStatus::values()),
        ]);

        $ticketSystem->update(['status' => $request->status]);

        return $this->apiResponseUpdated(new TicketSystemResource($ticketSystem));
    }

    public function destroy(TicketSystem $ticketSystem){
        $ticketSystem->ticketReplies()->each(function($ticketReply){
            $ticketReply->delete();
        });
        $ticketSystem->ticketAttachments()->each(function($ticketAttachment){
            $this->deleteMedia($ticketAttachment->file_path);
            $ticketAttachment->delete();
        });
        $ticketSystem->delete();
        return $this->apiResponseDeleted();
    }

    public function getReplies($ticket_id)
    {
        $ticket = TicketSystem::findOrFail($ticket_id);
        $replies = $ticket->ticketReplies()->with('user:id,name')->orderBy('created_at', 'desc')->get();

        return $this->apiResponseShow($replies);
    }

    public function storeReply(Request $request, $ticket_id)
    {
        $request->validate([
            'content' => 'required|string|max:10000000',
        ]);

        $reply = TicketReply::create([
            'ticket_id' => $ticket_id,
            'user_id' => auth()->id(),
            'message' => $request->content,
        ]);

        // Save to Firebase for real-time updates
        $messageData = [
            'id' => $reply->id,
            'ticket_id' => $ticket_id,
            'user_id' => auth()->id(),
            'user_name' => auth()->user()->name,
            'type'=>UserType::userType(UserType::from(auth()->user()->type)),
            'message' => $request->content,
            'created_at' => now()->toDateTimeString(),
        ];

        try {
            $this->database
                ->getReference("tickets/{$ticket_id}/messages")
                ->push($messageData);
        } catch (\Exception $e) {
            \Log::error('Failed to save message to Firebase', [
                'error' => $e->getMessage(),
                'ticket_id' => $ticket_id,
                'user_id' => auth()->id()
            ]);
        }

        return $this->apiResponseStored($reply->load('user:id,name'));
    }

    public function updateReply(Request $request, $ticket_id, $reply_id)
    {
        $request->validate([
            'content' => 'required|string|max:10000000',
        ]);

        $reply = TicketReply::where('id', $reply_id)
            ->where('ticket_id', $ticket_id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $reply->update([
            'message' => $request->content,
        ]);

        return $this->apiResponseUpdated($reply->load('user:id,name'));
    }

    public function destroyReply($ticket_id, $reply_id)
    {
        $reply = TicketReply::where('id', $reply_id)
            ->where('ticket_id', $ticket_id)
            ->where('user_id', auth()->id())
            ->firstOrFail();

        $reply->delete();
        return $this->apiResponseDeleted();
    }

    public function statistics()
    {
        $user= auth()->user();
        if($user->type == UserType::ADMIN->value && $user->role?->name == 'Super Admin'){
            $tickets = TicketSystem::with(['user', 'assignedTo', 'ticketReplies', 'ticketAttachments'])->get();
        }else{
            $tickets = TicketSystem::with(['user', 'assignedTo', 'ticketReplies', 'ticketAttachments'])->where('user_id', $user->id)->get();
        }

        $today = now();
        $lastWeek = now()->subWeek();
        $lastMonth = now()->subMonth();

        $totalTickets = $tickets->count();
        $closedTickets = $tickets->where('status', TicketStatus::CLOSED->value);
        $openTickets = $tickets->where('status', '!=', TicketStatus::CLOSED->value);

        $avgResolutionTime = $closedTickets->avg(function($ticket) {
            return $ticket->close_date && $ticket->open_date
                ? $ticket->open_date->diffInHours($ticket->close_date)
                : 0;
        });

        $statusCounts = [];
        foreach (TicketStatus::cases() as $status) {
            $statusCounts[strtolower($status->name)] = $tickets->where('status', $status->value)->count();
        }

        $priorityCounts = $tickets->groupBy('priority')->map->count();

        $recentlyOpened = $tickets->filter(function($ticket) use ($lastWeek) {
            return $ticket->open_date && $ticket->open_date >= $lastWeek;
        })->count();

        $recentlyClosed = $closedTickets->filter(function($ticket) use ($lastWeek) {
            return $ticket->close_date && $ticket->close_date >= $lastWeek;
        })->count();

        $responseTimes = $tickets->map(function($ticket) {
            $firstReply = $ticket->ticketReplies->sortBy('created_at')->first();
            return $firstReply && $ticket->created_at
                ? $ticket->created_at->diffInHours($firstReply->created_at)
                : null;
        })->filter();

        $avgResponseTime = $responseTimes->isNotEmpty() ? $responseTimes->avg() : 0;

        $ticketsWithReplies = $tickets->filter(function($ticket) {
            return $ticket->ticketReplies && $ticket->ticketReplies->isNotEmpty();
        });

        $ticketsWithoutReplies = $tickets->filter(function($ticket) {
            return !$ticket->ticketReplies || $ticket->ticketReplies->isEmpty();
        });

        $ticketsWithAttachments = $tickets->filter(function($ticket) {
            return $ticket->ticketAttachments && $ticket->ticketAttachments->isNotEmpty();
        });

        return $this->apiResponseShow([
            'summary' => [
                'total_tickets' => $totalTickets,
                'open_tickets' => $openTickets->count(),
                'closed_tickets' => $closedTickets->count(),
                'avg_resolution_time_hours' => round($avgResolutionTime, 2),
                'avg_response_time_hours' => round($avgResponseTime, 2),
            ],
            'by_status' => $statusCounts,
            'by_priority' => $priorityCounts,
            'recent_activity' => [
                'opened_last_week' => $recentlyOpened,
                'closed_last_week' => $recentlyClosed,
                'opened_last_month' => $tickets->filter(fn($t) => $t->open_date && $t->open_date >= $lastMonth)->count(),
                'closed_last_month' => $closedTickets->filter(fn($t) => $t->close_date && $t->close_date >= $lastMonth)->count(),
            ],
            'performance' => [
                'avg_replies_per_ticket' => $ticketsWithReplies->isNotEmpty()
                    ? $ticketsWithReplies->avg(fn($t) => $t->ticketReplies->count())
                    : 0,
                'tickets_without_reply' => $ticketsWithoutReplies->count(),
                'tickets_with_attachments' => $ticketsWithAttachments->count(),
            ],
            'time_metrics' => [
                'oldest_open_ticket_days' => $openTickets->isNotEmpty() && $openTickets->min('open_date')
                    ? $openTickets->min('open_date')->diffInDays($today)
                    : 0,
                'newest_ticket_hours' => $tickets->isNotEmpty() && $tickets->max('created_at')
                    ? $tickets->max('created_at')->diffInHours($today)
                    : 0,
            ]
        ]);
    }
}
