<?php

namespace App\Http\Controllers\Api\V1;

use App\Enums\TransactionTypeEnum;
use App\Http\Controllers\Controller;
use App\Http\Resources\Transaction\TransactionResource;
use App\Models\Transaction;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class Transaction2Controller extends Controller
{
    public function index()
    {
        return TransactionResource::collection(
            Transaction::with(['user', 'courseEnrollment.round', 'courseEnrollment.course', 'installment', 'acceptedBy'])->useFilters()->dynamicPaginate()
        );
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'user_id' => 'required|exists:users,id',
            'course_enro_id' => 'nullable|exists:course_enrollments,id',
            'installment_id' => 'nullable|exists:installments,id',
            'amount' => 'required|numeric',
            'payment_method' => 'required|string',
            'payment_date' => 'required|date',
            'type' => ['required', Rule::enum(TransactionTypeEnum::class)],
            'notes' => 'nullable|string',
            'accepted_by' => 'nullable|exists:users,id',
        ]);

        $transaction = Transaction::create($data);

        return $this->apiResponseStored(new TransactionResource($transaction->load(['user', 'courseEnrollment.round', 'courseEnrollment.course', 'installment', 'acceptedBy'])));
    }

    public function show(Transaction $transaction)
    {
        return $this->apiResponseShow(new TransactionResource($transaction->load(['user', 'courseEnrollment.round', 'courseEnrollment.course', 'installment', 'acceptedBy'])));
    }

    public function update(Request $request, Transaction $transaction)
    {
        $data = $request->validate([
            'amount' => 'sometimes|numeric',
            'payment_method' => 'sometimes|string',
            'payment_date' => 'sometimes|date',
            'type' => ['sometimes', Rule::enum(TransactionTypeEnum::class)],
            'notes' => 'nullable|string',
            'accepted_by' => 'nullable|exists:users,id',
        ]);

        $transaction->update($data);

        return $this->apiResponseUpdated(new TransactionResource($transaction->fresh()->load(['user', 'courseEnrollment.round', 'courseEnrollment.course', 'installment', 'acceptedBy'])));
    }
    public function destroy(Transaction $transaction)
    {
        $transaction->delete();

        return $this->apiResponseDeleted();
    }

    public function getTransactionsStatistics()
    {
        $now = now();

        $totalAmount = Transaction::sum('amount');
        $totalCount = Transaction::count();

        $thisMonthAmount = Transaction::whereMonth('payment_date', $now->month)
            ->whereYear('payment_date', $now->year)
            ->sum('amount');

        $lastMonthAmount = Transaction::whereMonth('payment_date', $now->subMonth()->month)
            ->whereYear('payment_date', $now->subMonth()->year)
            ->sum('amount');

        $maxTransaction = Transaction::max('amount');
        $avgTransaction = Transaction::avg('amount');

        $byType = Transaction::select('type')
            ->selectRaw('SUM(amount) as total_amount, COUNT(*) as total_transactions')
            ->groupBy('type')
            ->get()
            ->map(function ($row) {
                return [
                    'type' => $row->type->name ?? (string) $row->type,
                    'total_amount' => round($row->total_amount, 2),
                    'total_transactions' => $row->total_transactions,
                ];
            });

        return $this->apiResponseShow([
            'stats' => [
                'total_amount' => round($totalAmount, 2),
                'total_count' => $totalCount,
                'this_month_amount' => round($thisMonthAmount, 2),
                'last_month_amount' => round($lastMonthAmount, 2),
                'max_transaction' => round($maxTransaction, 2),
                'avg_transaction' => round($avgTransaction, 2),
            ],
            'by_type' => $byType,
        ]);
    }
}
