<?php

namespace App\Http\Controllers\Api\V1;

use App\Models\User;
use App\Models\Round;
use App\Models\Installment;
use App\Models\Transaction;
use App\Enums\PaymentStatus;
use App\Models\RoundStudent;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use App\Http\Requests\Transaction\StoreStudentTransactionRequest;
use App\Http\Resources\Transaction\TransactionResource;
use App\Http\Resources\User\UserResource;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Resources\Json\AnonymousResourceCollection;

class TransactionController extends Controller
{

    /**
     * Store a new transaction and related installments for a student in a round.
     *
     * @param StoreStudentTransactionRequest $request Validated request data.
     * @param Round $round The round for which the transaction is made.
     * @param User $user The student making the transaction.
     * @return JsonResponse API response with transaction data or error status.
     */
    public function storeStudentTransaction(StoreStudentTransactionRequest $request, Round $round, User $user): JsonResponse
    {
        $data         = $request->validated();
        //check if installments amounts sum is equal to total amount
        if (array_sum(array_column($data['installments'], 'amount')) != $data['total_amount']) {
            return $this->apiResponse([], 'total amount is not equal to installments amounts', 400);
        }
        $roundStudent = $this->validateAndGetRoundStudent($round, $user);
        if (!$roundStudent) {
            return $this->apiResponse([], __('http-statuses.404'), 404);
        }
        $paidAmount = $this->calculatePaidAmount($data['installments']);
        try {
            $transaction = DB::transaction(function () use ($data, $paidAmount, $roundStudent) {
                return $this->createTransactionWithInstallments($data, $paidAmount, $roundStudent);
            });

            return $this->apiResponseStored(TransactionResource::make($transaction));
        } catch (\Exception $e) {
            return $this->apiResponse([], __('http-statuses.500'), 500);
        }
    }

    /**
     * Validate if the user is a student in the given round and retrieve the RoundStudent record.
     *
     * @param Round $round The round to validate.
     * @param User $user The user to validate.
     * @return RoundStudent|null The RoundStudent record if valid, or null.
     */
    private function validateAndGetRoundStudent(Round $round, User $user): ?RoundStudent
    {
        abort_if(!$round->students->pluck('id')->contains($user->id), 404);

        return RoundStudent::where('round_id', $round->id)
            ->where('user_id', $user->id)
            ->first();
    }

    /**
     * Calculate the total paid amount from the provided installments.
     *
     * @param array $installments Array of installment data.
     * @return float The total paid amount.
     */
    private function calculatePaidAmount(array $installments): float
    {
        return array_sum(array_column(
            array_filter($installments, function ($installment) {
                return $installment['payment_status'] == PaymentStatus::PAID->value;
            }),
            'amount'
        ));
    }

    /**
     * Create a transaction and associate it with related installments and the RoundStudent record.
     *
     * @param array $data Validated transaction and installment data.
     * @param float $paidAmount The total paid amount for the transaction.
     * @param RoundStudent $roundStudent The RoundStudent record to associate with the transaction.
     * @return Transaction The created transaction record.
     */
    private function createTransactionWithInstallments(array $data, float $paidAmount, RoundStudent $roundStudent): Transaction
    {
        $transaction = Transaction::create([
            'total_amount'     => $data['total_amount'],
            'paid_amount'      => $paidAmount,
            'remaining_amount' => $data['total_amount'] - $paidAmount,
            'status'           => PaymentStatus::PENDING->value,
        ]);

        $data['installments'] = array_map(function ($installment) use ($transaction) {
            $installment['transaction_id'] = $transaction->id;
            return $installment;
        }, $data['installments']);

        $roundStudent->transaction_id = $transaction->id;
        $roundStudent->save();

        Installment::insert($data['installments']);
        $transaction->refresh();
        return $transaction->load('installments');
    }

    public function showStudentTransaction(Round $round, $userId): JsonResponse
    {
        abort_if(!$round->students->pluck('id')->contains($userId), 404);
        $roundStudentTransactionId = RoundStudent::where('round_id', $round->id)
            ->where('user_id', $userId)
            ->first()->transaction_id;
        if (!$roundStudentTransactionId) {
            return $this->apiResponseShow([
                'total_amount'     => $round->price,
                'paid_amount'      => 0,
                'remaining_amount' => $round->price,
                'installments'     => [],
            ]);
        }
        $roundStudentTransaction = Transaction::find($roundStudentTransactionId);
        $roundStudentTransaction->load('installments');
        return $this->apiResponseShow(new TransactionResource($roundStudentTransaction));
    }

    public function showAllTransactions(): AnonymousResourceCollection
    {
        $transactions = Transaction::useFilters()->with([
            'installments',
            'enrollment' => [
                'user:id,name,email',
                'round:id,name,course_id',
                'round.course:id,title',
            ],
        ])
            ->useFilters()
            ->latest()
            ->dynamicPaginate();
        return TransactionResource::collection($transactions);
    }

    public function ShowStudentsHasRemainingAmount()
    {
        $students = RoundStudent::query()
            ->whereHas('transaction.installments', function ($query) {
                $query->where('payment_status', PaymentStatus::PENDING->value);
            })
            ->with([
                'user:id,name,email',
                'round.course',
                'transaction.installments',
                // 'transaction'=>function($query){
                //     $query->useFilters();
                // }
            ])
            ->useFilters()
            ->latest()
            ->dynamicPaginate();
        return UserResource::collection($students);
    }

    public function updateInstallment(Installment $installment)
    {
        $data = request()->validate([
            'payment_status' => 'required',
        ]);
        $installment->update([
            'payment_status' => $data['payment_status'],
        ]);
        $installment->transaction->update([
            'paid_amount'      => $installment->transaction->paid_amount + $installment->amount,
            'remaining_amount' => $installment->transaction->remaining_amount - $installment->amount,
        ]);

        // Check if this is the last unpaid installment
        $hasRemainingUnpaidInstallments = $installment->transaction->hasUnpaidInstallment();

        if (!$hasRemainingUnpaidInstallments) {
            $installment->transaction->update([
                'status' => PaymentStatus::PAID->value
            ]);
        }

        return $this->apiResponseUpdated([]);
    }


    public function destroy(Transaction $transaction)
    {
        $transaction->delete();
        return $this->apiResponseDeleted();
    }
}
