<?php

namespace App\Http\Resources\User;

use App\Http\Resources\CourseEnrollment\CourseEnrollmentResource;
use App\Http\Resources\Evaluation\EvaluationResource;
use App\Http\Resources\NoticeAbsent\NoticeAbsentResource;
use App\Http\Resources\Role\RoleResource;
use App\Http\Resources\RoundStudent\RoundStudentResource;
use App\Http\Resources\TaskSubmission\TaskSubmissionResource;
use App\Http\Resources\Transaction\TransactionResource;
use Illuminate\Http\Request;
use App\Http\Traits\ResourceHelper;
use App\Models\Permission;
use Illuminate\Http\Resources\Json\JsonResource;
use App\Http\Resources\Role\PermissionResource;
use App\Http\Resources\Installment\InstallmentResource;
use App\Http\Resources\Lecture\LectureResource;
use App\Http\Resources\TaskFeedback\TaskFeedbackResource;
use App\Models\Lecture;

class UserResource extends JsonResource
{
    use ResourceHelper;

    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'name' => $this->name ?? null,
            'email' => $this->email ?? null,
            'phone' => $this->phone ?? null,
            'image' => $this->photo_path ?? null,
            'created_at' => $this->created_at?->format('Y-m-d H:i:s'),
            'status' => $this->status,
            'country' => $this->country,
            'gov' => $this->gov,
            'college' => $this->college,
            'gender'=>$this->gender,
            'bonus'=>$this->bonus,
            'desc_instructor'=>$this->desc_instructor,
            'role_id'=> $this->role_id,
            'assign_round'=>$this->assign_round,
            'google_id'=> $this->google_id,
            'apple_id'=> $this->apple_id,
            'is_open'=> $this->is_open,
            'last_activity'=> $this->last_activity?->format('Y-m-d H:i:s'),
            'last_logout_at'=> $this->last_logout_at?->format('Y-m-d H:i:s'),
            'is_live'=>$this->isLive(),
            'attendance_rate' => $this->round_id !== null ? $this->getAttendanceRateForRound($this->round_id,$this->id) : null,
            'task_submission_rate' => $this->round_id !== null ? $this->getTaskSubmissionRateForRound($this->round_id,$this->id) : null,
            'average_rating' => null,
            'role' => $this->whenLoaded('role', function () {
                return [
                    'role_name'=>$this->role->name,
                    'permissions' => $this->role->permissions->map(function ($permission) {
                        return [
                            'id' => $permission->id,
                        ];
                    }),
                ];
            }),
            'birthdate' => $this->birthdate,
            'placement_test_status' => $this->placement_test_status,

            'enrolled_courses' => $this->whenLoaded('enrolledCourses',
                fn() => CourseEnrollmentResource::collection($this->enrolledCourses)
            ),

            'student_rounds' => $this->whenLoaded('studentRounds', function () {
                return $this->studentRounds->map(function ($round) {
                    return [
                        'id' => $round->id,
                        'name' => $round->name,
                        'status' => $round->pivot->status,
                    ];
                });
            }),

            'transactions' => $this->whenLoaded('transactions', function() {
                return TransactionResource::collection($this->transactions);
            }),

            'installments' => $this->whenLoaded('installments', function() {
                return InstallmentResource::collection(
                    $this->installments
                );
            }),

            'my_evaluation' => $this->whenLoaded('evaluation',
                fn() => new EvaluationResource($this->evaluation)
            ),

            'task_submissions' => $this->whenLoaded('taskSubmissions',
                   fn() => new TaskSubmissionResource($this->taskSubmissions->first()?->load('feedbacks'))
            ),
           'attendance_evaluation' => $this->whenLoaded('evaluations', function () {
                $attended = $this->evaluations->where('is_attend', true)->count();
                // $total = $this->ro->count();

                return [
                    'attended' => $attended,
                    // 'total' => $total,
                    // 'rate' => $total > 0 ? round(($attended / $total) * 100, 2) : 0,
                    'text' => $attended,
                ];
            }),

            'task_evaluation_average' => $this->whenLoaded('taskSubmissions', function () {
                if ($this->taskSubmissions->isEmpty()) return null;

                $average = $this->taskSubmissions->avg('task_degree');
                return round($average, 2);
            }),

            'enrolled_rounds' => $this->whenLoaded('enrolledRounds', function () {
                return $this->enrolledRounds->map(function ($round) {
                    return [
                        'id' => $round->id,
                        'name' => $round->name,
                        'status'=>$round->status,
                        'lectures' => LectureResource::collection($round->lectures->load('taskSubmissions')),
                    ];
                });
            }),
            'task_feedbacks' => $this->whenLoaded('taskFeedbacks', function () {
                return TaskFeedbackResource::collection($this->taskFeedbacks);
            }),

            'notices' => $this->whenLoaded('notices',function(){
                return $this->notices->map(function ($notice) {
                    return [
                        'id' => $notice->id,
                        'content'=>$notice->content,
                        'creator_name'=>$notice->creator_name,
                        'updater_name'=>$notice->updater_name,
                    ];
                });
            }),
            'upcoming_installments'=>$this->whenLoaded('installments',function(){
                return InstallmentResource::collection($this->installments()->where('due_date', '<=', now()->addDays(3))->get());
            }),
            'absences_notices'=> $this->whenLoaded('absencesNotices', function () {
                return NoticeAbsentResource::collection($this->absencesNotices);
            }),
            'skills'=>$this->whenLoaded('skills', function () {
                return $this->skills;
            }),
        ];
    }
}
