<?php

namespace App\Http\Traits;

use App\Enums\UserType;
use App\Models\Notification;
use App\Models\RoundStudent;
use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification as FirebaseNotification;
use function PHPUnit\Framework\isEmpty;

trait NotificationTrait
{
    use ApiResponse;

    public function sendPushNotificationClientSide(
        string $type,
        string $action,
        ?string $action_id = null,
    ) {
        if($type==="client_side"){
            $data = $this->getNotificationData($type, $action, $action_id);
            $this->prepareNotificationData($data, $type, $action, $action_id);
            return $this->handleClientSideNotification($data);
        }
        $data = $this->getNotificationData($type, $action, $action_id);
        $this->prepareNotificationData($data, $type, $action, $action_id);
        return $this->handleAdminNotification($data,$action_id);
    }

    private function handleAdminNotification(array $data,$action_id)
    {
        $users = $this->getNotifiablesByContext($action_id);

        if (isEmpty($users)) {
            return $this->apiResponseFailed(__('main.notifications.errors.no_users_found'));
        }

        $this->storeNotification($data, $users);

        $deviceTokens = $users->pluck('device_token')
            ->filter()
            ->toArray();
        $userIds = $users->pluck('id')
            ->filter()
            ->toArray();

        $this->sendToUsersViaFirebase($userIds,$data);

        $this->sendToMultipleDevices($deviceTokens,$data);

        return $this->apiResponseShow(
            ['message' => 'Client notification stored successfully'],
            __('main.notifications.success.stored')
        );
    }

    private function handleClientSideNotification(array $data)
    {
        $users = User::where('type',UserType::ADMIN)->get();

        if (!$users) {
            return $this->apiResponseFailed(__('main.notifications.errors.no_users_found'));
        }

        $this->storeNotification($data, $users);

        $this->saveToFirebaseRealtimeDB($data);

        return $this->apiResponseShow(
            ['message' => 'Client notification stored successfully'],
            __('main.notifications.success.stored')
        );
    }

    // private function getTargetUsers($user_id, $round_id, $course_id)
    // {
    //     if ($round_id || $course_id) {
    //         return $this->getNotifiablesByContext($round_id, $course_id);
    //     }

    //     return User::where('id', $user_id)->get();
    // }

    private function getNotifiablesByContext($action_id)
    {
        $users=[];
        if ($action_id) {
            $users=RoundStudent::where('round_id',$action_id)->pluck('user_id')->toArray();
            $users=User::whereIn('id',$users)->get();
        }

        return $users;
    }

    private function getDeviceTokens($users)
    {
        return $users->pluck('device_token')
            ->filter()
            ->unique()
            ->values()
            ->toArray();
    }

    private function prepareNotificationData(
        array $data,
        string $type,
        string $action,
        $action_id
    ): array {
        return [
            'user_id'=>$data['user_id']??null,
            'title' => $data['title'],
            'content' => $data['content'],
            'action'=> $data['action'],
            'action_id' => $action_id,
            'data' => [
                'type' => $type,
                'action' => $action,
                'timestamp' => now()->toISOString()
            ]
        ];
    }

    public function getNotificationData(string $type, string $action, ?string $action_id = null): array
    {
        $data = [
            'user_id'=>null,
            'title' => '',
            'content' => '',
            'action' => $action,
            'action_id' => $action_id,
        ];

        if($type==="client_side"){
            $this->setClientSideNotificationContent($data, $action);
        }else{
            $this->setAdminNotificationContent($data, $action);
        }

        return $data;
    }


    private function setClientSideNotificationContent(array &$data, string $action): void
    {
        $locale = app()->getLocale();
        $prettyAction = ucfirst(str_replace('_', ' ', $action));

        $messages = [
            'en' => [
                'title' => "Notification: $prettyAction",
                'content' => "You have a new action: $prettyAction",
            ],
            'ar' => [
                'title' => "إشعار: $prettyAction",
                'content' => "لديك إجراء جديد: $prettyAction",
            ],
        ];

        $data['title'] = $messages[$locale]['title'] ?? $messages['en']['title'];
        $data['content'] = $messages[$locale]['content'] ?? $messages['en']['content'];
    }

    private function setAdminNotificationContent(array &$data, string $action): void
    {
        $locale = app()->getLocale();

        $messages = [
            'add_course' => [
                'en' => [
                    'title' => 'Admin added a course',
                    'content' => 'The admin has added a new course to your account.',
                ],
                'ar' => [
                    'title' => 'قام المسؤول بإضافة دورة',
                    'content' => 'قام المسؤول بإضافة دورة جديدة إلى حسابك.',
                ],
            ],
            'add_lecture' => [
                'en' => [
                    'title' => 'Admin added a lecture',
                    'content' => 'A new lecture has been added to your course.',
                ],
                'ar' => [
                    'title' => 'قام المسؤول بإضافة محاضرة',
                    'content' => 'تمت إضافة محاضرة جديدة إلى دورتك.',
                ],
            ],
            'assign_homework' => [
                'en' => [
                    'title' => 'Homework assigned',
                    'content' => 'The admin has assigned new homework to you.',
                ],
                'ar' => [
                    'title' => 'تم تعيين واجب منزلي',
                    'content' => 'قام المسؤول بتعيين واجب منزلي جديد لك.',
                ],
            ],
            'enroll_student' => [
                'en' => [
                    'title' => 'You have been enrolled',
                    'content' => 'The admin has enrolled you in a course.',
                ],
                'ar' => [
                    'title' => 'تم تسجيلك في الدورة',
                    'content' => 'قام المسؤول بتسجيلك في دورة.',
                ],
            ],
            'add_quiz' => [
                'en' => [
                    'title' => 'Admin added a quiz',
                    'content' => 'A new quiz has been added to your course.',
                ],
                'ar' => [
                    'title' => 'قام المسؤول بإضافة اختبار',
                    'content' => 'تمت إضافة اختبار جديد إلى دورتك.',
                ],
            ],
        ];

        $defaultTitle = ucfirst(str_replace('_', ' ', $action));

        $data['title'] = $messages[$action][$locale]['title'] ?? "Admin action: $defaultTitle";
        $data['content'] = $messages[$action][$locale]['content'] ?? "The admin performed a new action: $defaultTitle.";
    }

    public function storeNotification(array $data, $notifiables): void
    {
        $notifications = $notifiables->map(function ($notifiable) use ($data) {
            $notification = new Notification($this->prepareNotificationModel($data, $notifiable));
            $notification->id = \Illuminate\Support\Str::uuid();
                $notification->save();
        });
    }
    public function storeNotificationSingle(array $data, $notifiable): void
    {
        $notification = new Notification($this->prepareNotificationModel($data, $notifiable));
            $notification->id = \Illuminate\Support\Str::uuid();
                $notification->save();
    }
    private function prepareNotificationModel(array $data, $notifiable): array
    {
        return [
            'id' => \Illuminate\Support\Str::uuid(),
            'data' => $data,
            'notifiable_id' => $notifiable->id,
            'notifiable_type' => get_class($notifiable),
            'type' => 'notification',
            'read_at' => null,
            'created_at' => now(),
            'updated_at' => now(),
        ];
    }
    public function sendToMultipleDevices(array $tokens, array $data): ?object
    {
        try {
            if (empty($tokens)) {
                return null;
            }

            $messaging = $this->getFirebaseMessaging();

            $notification = FirebaseNotification::create($data['title'], $data['content']);

            $message = CloudMessage::new()
                ->withNotification($notification)
                ->withData($data ?? []);

            $pushResult = $messaging->sendMulticast($message, $tokens);

            Log::info('FCM multicast results', [
                'tokens' => count($tokens),
                'successes' => $pushResult->successes()->count(),
                'failures' => $pushResult->failures()->count()
            ]);

            return $pushResult;

        } catch (\Exception $e) {
            Log::error('Firebase operation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return null;
        }
    }

    public function sendToUsersViaFirebase(array $userIds, array $notificationData): bool
    {
        try {
            if (empty($userIds)) {
                Log::warning('No user IDs provided for Firebase notification');
                return false;
            }

            $database = $this->getFirebaseDatabase();
            $successCount = 0;

            foreach ($userIds as $userId) {
                try {
                    // Create user-specific notification path
                    $userNotificationsRef = $database->getReference("user_notifications/$userId")->push();

                    $notificationPayload = [
                        'id' => Str::uuid()->toString(),
                        'user_id' => $userId,
                        'title' => $notificationData['title'] ?? 'New Notification',
                        'content' => $notificationData['content'] ?? '',
                        'data' => $notificationData['data'] ?? [],
                        'created_at' => time(),
                        'is_read' => false,
                        'action' => $notificationData['action'] ?? null,
                        'action_id' => $notificationData['action_id'] ?? null
                    ];

                    $userNotificationsRef->set($notificationPayload);
                    $successCount++;

                    Log::debug('Notification saved to Firebase', [
                        'user_id' => $userId,
                        'notification_id' => $userNotificationsRef->getKey()
                    ]);

                } catch (\Exception $e) {
                    Log::error('Failed to save notification for user', [
                        'user_id' => $userId,
                        'error' => $e->getMessage()
                    ]);
                    continue;
                }
            }

            Log::info('Firebase notifications completed', [
                'total_users' => count($userIds),
                'successful' => $successCount,
                'failed' => count($userIds) - $successCount
            ]);

            return $successCount > 0;

        } catch (\Exception $e) {
            Log::error('Firebase operation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }

    /**
     * Get Firebase Messaging instance
     */
    protected function getFirebaseMessaging()
    {
        return (new Factory)
            ->withServiceAccount(config('services.firebase.credentials'))
            ->createMessaging();
    }

    protected function getFirebaseDatabase()
    {
        return (new Factory)
            ->withServiceAccount(config('services.firebase.credentials'))
            ->withDatabaseUri(config('services.firebase.database_url'))
            ->createDatabase();
    }
    protected function saveToFirebaseRealtimeDB(array $data): bool
    {
        try {
            $database = $this->getFirebaseDatabase();

            $notificationRef = $database->getReference('admin_notifications')->push();

            $notificationData = [
                'title' => $data['title'],
                'content' => $data['content'],
                'action' => $data['action'] ?? 'general',
                'action_id' => $data['action_id'] ?? '',
                'created_at' => time(),
                'url' => $data['url'] ?? null,
                'is_read' => false,
                'metadata' => $data['data'] ?? []
            ];

            $notificationRef->set($notificationData);

            return true;
        } catch (\Exception $e) {
            Log::error('Failed to save to Firebase Realtime DB', [
                'error' => $e->getMessage(),
                'data' => $data
            ]);
            return false;
        }
    }

}
