<?php

namespace App\Models;

use App\Enums\DefineStatus;
use App\Enums\PaymentStatus;
use App\Filters\CourseFilters;
use Essa\APIToolKit\Filters\Filterable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class Course extends Model
{
    use HasFactory, Filterable;

    protected string $default_filters = CourseFilters::class;

    protected $casts = [
        'status'              => DefineStatus::class,
        'what_you_will_learn' => 'array',
        'course_content'=>'array',
        'skills'=>'array'
    ];

    protected $fillable = [
        'category_id',
        'title',
        'image',
        'price',
        'description',
        'preview_video',
        'preview_video_thumbnail',
        'status',
        'sessions_count',
        'duration',
        'what_you_will_learn',
        'learning_plan',
        'topic_id',
        'placement_test_available',
        'course_content',
        'skills'
    ];

    protected $appends = ['is_enrolled'];

    public function category(): BelongsTo
    {
        return $this->belongsTo(Category::class);
    }

    public function skills(): HasMany
    {
        return $this->hasMany(Skill::class);
    }

    public function topic()
    {
        return $this->belongsTo(Topic::class);
    }

    public function lectures()
    {
        return $this->hasManyThrough(Lecture::class, Round::class, 'course_id', 'round_id');
    }

    /**
     * Get the enrollment status for the authenticated user
     *
     * @return bool
     */
    public function getIsEnrolledAttribute(): bool
    {
        if (!auth()->check()) {
            return false;
        }

        return CourseEnrollment::isUserEnrolled(auth()->id(), $this->id);
    }

    /**
     * Get enrollments for this course
     */
    public function enrollments(): HasMany
    {
        return $this->hasMany(CourseEnrollment::class);
    }

    public function getImageAttribute(): ?string
    {
        if (!isset($this->attributes['image'])) {
            return null;
        }

        $image = $this->attributes['image'];

        // Check if it's already a URL (starts with http or https)
        if (preg_match('/^https?:\/\//', $image)) {
            return $image;
        }

        return asset(Storage::url($image));
    }

    public function getPreviewVideoThumbnailAttribute(): ?string
    {
        if (!isset($this->attributes['preview_video_thumbnail'])) {
            return null;
        }

        $thumbnail = $this->attributes['preview_video_thumbnail'];

        // Check if it's already a URL (starts with http or https)
        if (preg_match('/^https?:\/\//', $thumbnail)) {
            return $thumbnail;
        }

        return asset(Storage::url($thumbnail));
    }

    public function getLearningPlanAttribute(): ?string
    {
        if (!isset($this->attributes['learning_plan'])) {
            return '';
        }

        $learningPlan = $this->attributes['learning_plan'];

        if (preg_match('/^https?:\/\//', $learningPlan)) {
            return $learningPlan;
        }

        return asset(Storage::url($learningPlan));
    }

    public function rounds(): HasMany
    {
        return $this->hasMany(Round::class);
    }
    public function userRounds()
    {
        if (!auth()->user()) {
            return 0;
        }
        $user              = auth()->user()->id;
        $thisCourseGroupId = $this->rounds()->pluck('id');
        return RoundStudent::where('user_id', $user)
            ->whereIn('round_id', $thisCourseGroupId)
            ->where('status', DefineStatus::ACTIVE->value)
            ->where('payment_status', PaymentStatus::PAID->value)
            ->first('id');
    }


    public function freeMaterials(){
        return FreeMaterial::whereJsonContains('course_ids', $this->id)->get();
    }

}

