<?php

namespace App\Models;

use Carbon\Carbon;
use App\Enums\DefineStatus;
use App\Filters\LectureFilters;
use Essa\APIToolKit\Filters\Filterable;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Factories\HasFactory;


class Lecture extends Model
{
    use HasFactory, Filterable;

    protected string $default_filters = LectureFilters::class;

    protected $fillable = [
        'round_id',
        'name',
        'url',
        'start_time',
        'week_number',
        'status',
    ];

    /**
     * Relationship with round
     * The lecture is belongs to a round
     */
    public function round(): BelongsTo
    {
        return $this->belongsTo(Round::class);
    }

    /**
     * Relationship with evaluations
     * The lecture can have many evaluations
     */
    public function evaluations(): HasMany
    {
        return $this->hasMany(Evaluation::class);
    }

    /**
     * Relationship with zoom meetings
     * The lecture can have many meetings
     */
    public function zoomMeetings()
    {
        return $this->hasOne(ZoomMeeting::class, 'lecture_id', 'id');
    }

    public function taskSubmissions()
    {
        return $this->hasMany(TaskSubmission::class);
    }

    public function taskSubmissionsIsCorrected()
    {
        if (!$this->relationLoaded('round') || !$this->round->relationLoaded('students')) {
            return null;
        }

        $studentIds = $this->round->students->pluck('id');
        $query = $this->taskSubmissions()->whereIn('user_id', $studentIds);

        $correctedCount = (clone $query)->where('task_degree', '>=', 0)->count();
        $totalCount = $query->count();

        return $correctedCount . '/' . $studentIds->count();
    }




    /**
     * Accessor that can access the main image
     */
    public function getTaskAttribute($value)
    {
        if (filter_var($value, FILTER_VALIDATE_URL)) {
            return $value;
        }

          // Otherwise, generate the asset URL
        return $value ? asset('storage/' . $value): null;
    }

    public function getDateAttribute($value)
    {
        return Carbon::parse($value)->format('Y-m-d');
    }


    public function scopeActive($query)
    {
        return $query->where('status', DefineStatus::ACTIVE);
    }

    public function studentTaskSubmission(){
        return $this->hasOne(TaskSubmission::class)->where('user_id', auth()->id());
    }
    public function studentEvaluation(){
        return $this->hasOne(Evaluation::class)->where('user_id', auth()->id());
    }
    public function attachments()
    {
        return $this->morphMany(Attachment::class, 'attachable');
    }

    public function questions(){
        return $this->hasMany(LectureQuestion::class);
    }

    public function getIsTodayAttribute()
    {
        return false;
    }


    public function lectureTasks(){
        return $this->hasOne(LectureTask::class);
    }

    public function lectureVideos(){
        return $this->hasMany(LectureVideo::class);
    }

}
