<?php

namespace App\Models;

use App\Enums\UserType;
use App\Filters\PayoutFilters;
use Essa\APIToolKit\Filters\Filterable;
use Illuminate\Database\Eloquent\Model;
use Spatie\Activitylog\Traits\LogsActivity;
use Spatie\Activitylog\LogOptions;

class Payout extends Model
{
    use Filterable;

    protected string $default_filters = PayoutFilters::class;

    /**
     * Mass-assignable attributes.
     *
     * @var array
     */
    protected $fillable = [
        'user_id',
        'admin_id',
        'amount',
        'payment_reference',
        'note',
        'transfer_type',
    ];

    public function getActivitylogOptions(): LogOptions
    {
        return LogOptions::defaults()
            ->logFillable()
            ->logOnlyDirty();
    }

    /**
     * Get the instructor who received the payout.
     */
    public function instructor()
    {
        return $this->belongsTo(User::class, 'user_id')->where('type', UserType::INSTRUCTOR->value);
    }

    /**
     * Get the admin who made the payout.
     */
    public function admin()
    {
        return $this->belongsTo(User::class, 'admin_id')->where('type', UserType::ADMIN->value);
    }


    public function getPaymentReferencePathAttribute()
    {
        return $this->attributes['payment_reference'] ?? null;
    }

    /**
     * Accessor that can access the main image
     */
    public function getPaymentReferenceAttribute($value)
    {
        if (filter_var($value, FILTER_VALIDATE_URL)) {
            return $value;
        }
    
        // Otherwise, generate the asset URL
        return $value ? asset('storage/' . $value) : null;
    }
}
