<?php

namespace App\Models;


use App\Enums\FeedbackType;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Storage;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Support\Facades\Auth;
use App\Filters\TaskFeedbackFilters;
use Essa\APIToolKit\Filters\Filterable;

class TaskFeedback extends Model
{
    use HasFactory, Filterable;

    protected string $default_filters = TaskFeedbackFilters::class;

    protected $table = 'task_feedbacks';
    protected $casts = [
        'type' => FeedbackType::class,
    ];

    /**
     * Mass-assignable attributes.
     *
     * @var array
     */
    protected $fillable = [
        'task_submission_id',
        'content',
        'type',
        'note',
        'created_by',
        'read'
    ];

    public static function boot()
    {
        parent::boot();

        static::creating(function ($model) {
            if (Auth::check()) {
                $model->created_by = Auth::id();
            }
        });
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function getCreatedByUserAttribute()
    {
        return $this->relationLoaded('createdBy') ? $this->createdBy?->name : null;
    }

    public function taskSubmission()
    {
        return $this->belongsTo(TaskSubmission::class);
    }

    public function getContentAttribute($value)
    {
        if($this->type->value == FeedbackType::IMAGE->value || $this->type->value == FeedbackType::AUDIO->value){
            return asset(Storage::url($value));
        }
        return $value;
    }

    public function lecture()
    {
        return $this->hasOneThrough(Lecture::class, TaskSubmission::class, 'id', 'id', 'task_submission_id', 'lecture_id');
    }
}
