<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;

use App\Enums\PaymentStatus;
use App\Enums\UserType;
use App\Enums\DefineStatus;
use App\Filters\UserFilters;
use Laravel\Sanctum\HasApiTokens;
use Essa\APIToolKit\Filters\Filterable;
use Illuminate\Notifications\Notifiable;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Builder;
use App\Models\ProfileLink;
use App\Models\Project;
// use Spatie\Permission\Traits\HasRoles;

class User extends Authenticatable
{
    use HasFactory, Notifiable, HasApiTokens, Filterable;

    protected string $default_filters = UserFilters::class;

    protected $casts = [
        'status' => DefineStatus::class,
        'birthdate'=>'date',
        'is_open'=>'boolean',
        'last_logout_at'=>'datetime',
        'last_activity'=>'datetime',
        'assign_round'=>'array',
        'skills_status'=>'boolean'
    ];

    /**
     * The attributes that are mass assignable.
     *
     * @var list<string>
     */
    protected $fillable = [
        'name',
        'phone',
        'photo',
        'type',
        'status',
        'email',
        'password',
        'device_token',
        'placement_test_status',
        'country',
        'gov',
        'college',
        'birthdate',
        'role_id',
        'google_id',
        'apple_id',
        'gender',
        'desc_instructor',
        'updated_at',
        'last_activity',
        'is_open',
        'last_logout_at',
        'assign_round',
        'skills_status',
        'bonus'
    ];

    public function isLive()
    {
        return $this->last_activity && $this->last_activity->gt(now()->subMinutes(3));
    }

    public static function boot(){
        parent::boot();
        static::creating(function ($model) {
            $model->placement_test_status = 1;
        });
    }

    public function notices(){
        return $this->hasMany(NoticeStudent::class,'user_id');
    }

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * Get the attributes that should be cast.
     *
     * @return array<string, string>
     */
    protected function casts(): array
    {
        return [
            'email_verified_at' => 'datetime',
            'password'          => 'hashed',
            'last_logout_at'    =>'datetime',
            'last_activity'=>'datetime',
            'assign_round'=>'array'
        ];
    }

    public function role(){
        return $this->belongsTo(Role::class, 'role_id');
    }

    public function round(): HasMany
    {
        return $this->hasMany(Round::class);
    }

    public function evaluation()
    {
        return $this->hasOne(Evaluation::class);
    }


    public function tickets(): HasMany
    {
        return $this->hasMany(Ticket::class);
    }

    public function absencesNotices(){
        return $this->hasMany(NoticeAbsent::class,'student_id');
    }

    /**
     * Relationship with rounds
     * (as a student, Many to Many)
     */
    public function studentRounds(): BelongsToMany
    {
        return $this->belongsToMany(Round::class, 'round_students')
                    ->withPivot('status', 'transaction_id', 'total_amount', 'enrollment_date', 'payment_method', 'payment_status')
                    ->withTimestamps();
    }

    /**
     * Relationship with rounds
     * (as a supervisor, Many to Many)
     */
    public function supervisorRounds(): BelongsToMany
    {
        return $this->belongsToMany(Round::class, 'round_supervisor')
                    ->withPivot('status')
                    ->withTimestamps();
    }

    public function taskSubmissions()
    {
        return $this->hasMany(TaskSubmission::class);
    }

    public function taskFeedbacks()
    {
        return $this->hasManyThrough(
            TaskFeedback::class,
            TaskSubmission::class,
            'user_id',
            'task_submission_id',
            'id',
            'id'
        );
    }



    // to use in display the top 3 students score
    public function evaluations()
    {
        return $this->hasMany(Evaluation::class);
    }

    /**
     * User type scopes
     */
    public function scopeSupervisors(Builder $query): Builder
    {
        return $query->where('type', UserType::SUPERVISOR);
    }

    public function scopeStudents(Builder $query): Builder
    {
        return $query->where('type', UserType::STUDENT);
    }

    public function scopeInstructors(Builder $query): Builder
    {
        return $query->where('type', UserType::INSTRUCTOR);
    }

    public function scopeAdmins(Builder $query): Builder
    {
        return $query->where('type', UserType::ADMIN);
    }


    public function getPhotoPathAttribute(): ?string
    {
        $photo = $this->attributes['photo'] ?? null;
        if (filter_var($photo, FILTER_VALIDATE_URL)) {
            return $photo;
        }
        return $photo !== null ? asset('storage/' . $photo) : null;
    }

    /**
     * Accessor that can access the user photo
     */
    public function getPhotoAttribute($value)
    {
        if (filter_var($value, FILTER_VALIDATE_URL)) {
            return $value;
        }

        // Otherwise, generate the asset URL
        return $value ? asset('storage/' . $value) : null;
    }

    // public function enrolledRounds()
    // {
    //     return $this->hasManyThrough(Round::class, RoundStudent::class, 'user_id', 'id', 'id', 'round_id')->withPivot('created_at');
    // }

    public function enrolledRounds()
    {
        return $this->belongsToMany(Round::class, 'round_students', 'user_id', 'round_id')
            ->withPivot('created_at', 'total_amount', 'enrollment_date', 'status', 'payment_method', 'payment_status', 'transaction_id');
    }

    public function getTotalPoints($roundId)
    {
        return RoundStudent::where('round_id', $roundId)->where('user_id', $this->id)->first()->total_points;
    }

    public function getTotalTaskDegreesForRound(Round $round): int
    {
        return $this->taskSubmissions()
            ->whereHas('lecture', function ($query) use ($round) {
                $query->where('round_id', $round->id);
            })
            ->sum('task_degree');
    }

    public function enrolledCourses()
    {
        return $this->hasMany(CourseEnrollment::class, 'user_id', 'id');
    }

    public function notifications()
    {
        return $this->morphMany(Notification::class, 'notifiable');
    }

    public function roundHasEnded($course_id)
    {
        $user = auth()->user();

        if (!$user) {
            return false;
        }

        $activeRoundIds = Round::where('course_id', $course_id)
            ->where('status', DefineStatus::ACTIVE->value)
            ->where('start_date', '<=', now())
            ->where('end_date', '>=', now())
            ->pluck('id');

        if ($activeRoundIds->isEmpty()) {
            return false;
        }

        return RoundStudent::where('user_id', $user->id)
            ->whereIn('round_id', $activeRoundIds)
            ->where('status', DefineStatus::ACTIVE->value)
            ->where('payment_status', PaymentStatus::PAID->value)
            ->exists();
    }

    public function transactions(){
        return $this->hasMany(Transaction::class, 'user_id', 'id');
    }

    public function installments(){
        return $this->hasMany(Installment::class,'user_id');
    }


    public function getAttendanceRateForRound($roundId,$user_id)
    {
        $round=Round::find($roundId);
        $lectureIds = $round->lectures()->pluck('lectures.id');

        $totalLectures = $lectureIds->count();
        if ($totalLectures === 0) return 0;

        $actualAttendances = Evaluation::whereIn('lecture_id', $lectureIds)
            ->where('user_id', $user_id)
            ->where('is_attend', true)
            ->count();

        return round(($actualAttendances / $totalLectures) * 100, 2);
    }

    public function courseEnrollments()
    {
        return $this->hasMany(CourseEnrollment::class,'user_id','id');
    }

    public function getTaskSubmissionRateForRound($roundId,$user_id)
    {
        $round=Round::find($roundId);
        $lectureIds = $round->lectures()->pluck('lectures.id');
        $taskSubmissions = TaskSubmission::where('user_id',$user_id)->whereIn('lecture_id', $lectureIds)->get();
        $totalEvaluations = $taskSubmissions->sum('task_degree');
        $lectureCount=($lectureIds->count()*10);

        return round(($totalEvaluations / $lectureCount) * 100, 2);
    }

    public function profileLink()
    {
        return $this->hasOne(ProfileLink::class);
    }

    public function projects()
    {
        return $this->hasMany(Project::class);
    }
    public function skills()
    {
        return $this->hasMany(Skill::class, 'user_id', 'id');
    }

    // public function getAverageRatingForRound($roundId): ?float
    // {
    //     $round=Round::find($roundId);
    //     $lectureIds = $round->lectures()->pluck('lectures.id');

    //     if ($lectureIds->isEmpty()) return null;

    //     $evaluations = Evaluation::whereIn('lecture_id', $lectureIds)
    //         ->where('user_id', $this->id)
    //         ->get();

    //     if ($evaluations->isEmpty()) return null;

    //     $totalEvaluations = $evaluations->count();
    //     $attendedCount = $evaluations->where('is_attend', true)->count();

    //     return round(($attendedCount / $totalEvaluations) * 100, 2);
    // }

}
