<?php

namespace App\Services;

use App\Enums\DefineStatus;
use App\Enums\PaymentStatus;
use App\Enums\TypeCoupon;
use App\Models\Course;
use App\Models\Coupon;
use App\Models\CourseEnrollment;
use App\Interfaces\PaymentGatewayInterface;
use App\Models\Round;
use App\Models\RoundStudent;

class CourseEnrollmentService
{
    public function __construct(
        protected PaymentGatewayInterface $paymentGateway
    ) {}

    public function handleEnrollment(array $data, int $userId): array
    {
        $course = Course::findOrFail($data['course_id']);

        $this->validatePreviousEnrollment($userId, $data['course_id']);

        if (!empty($data['coupon_code'])) {
            $data = $this->applyCoupon($data, $course);
        } else {
            $data['total_amount'] = $course->price;
        }
        $enrollment = $this->createEnrollment($data);
        return $this->processPayment($enrollment, $course->title);
    }

    private function validatePreviousEnrollment(int $userId, int $courseId): void
    {
        $previousEnrollment = CourseEnrollment::where('user_id', $userId)
            ->where('course_id', $courseId)
            ->first();

        if (!$previousEnrollment) {
            return;
        }

        if ($this->shouldDeleteUnpaidEnrollment($previousEnrollment)) {
            $previousEnrollment->delete();
            return;
        }

        if ($this->isFullyEnrolled($previousEnrollment)) {
            $this->cleanupActiveRoundsForUser($userId, $courseId);
        }
    }

    private function shouldDeleteUnpaidEnrollment(CourseEnrollment $enrollment): bool
    {
        return !$enrollment->status && !$enrollment->payment_status;
    }

    private function isFullyEnrolled(CourseEnrollment $enrollment): bool
    {
        return $enrollment->status && $enrollment->payment_status;
    }

    private function cleanupActiveRoundsForUser(int $userId, int $courseId): void
    {
        $activeRoundIds = Round::query()
            ->forCourse($courseId)
            ->active()
            ->currentlyRunning()
            ->pluck('id');

        RoundStudent::query()
            ->forUser($userId)
            ->forRounds($activeRoundIds)
            ->activeAndPaid()
            ->delete();
    }

    private function applyCoupon(array $data, Course $course): array
    {
        $coupon = Coupon::where('code', $data['coupon_code'])
            ->where('status', true)
            ->where('end_date', '>', now())
            ->whereRaw('uses < max_uses')
            ->first();

        if (!$coupon) {
            throw new \Exception(__('main.invalid_coupon'));
        }

        $type = TypeCoupon::from($coupon->type);

        $discountAmount = match ($type) {
            TypeCoupon::DISCOUNT => ($course->price * $coupon->discount_percentage) / 100,
            TypeCoupon::FIXED => $coupon->discount_amount,
        };

        $data['total_amount'] = max($course->price - $discountAmount, 0);
        $data['coupon_id'] = $coupon->id;

        $coupon->increment('uses');

        return $data;
    }

    private function createEnrollment(array $data): CourseEnrollment
    {
        return CourseEnrollment::create($data);
    }

    private function processPayment(CourseEnrollment $enrollment, string $courseTitle): array
    {
        $paymentResponse = $this->initiatePayment($enrollment->total_amount, $courseTitle);
        $paymentResponseData = json_decode($paymentResponse->getContent(), true);

        if (!$paymentResponseData['success']) {
            throw new \Exception($paymentResponseData['message']);
        }

        $enrollment->update([
            'payment_status' => false,
            'transaction_id' => $paymentResponseData['data']['data']['intention_order_id'],
        ]);

        return [
            'success' => true,
            'data' => [
                'redirect_url' => $paymentResponseData['data']['redirect_url'],
                'amount' => $enrollment->total_amount,
            ],
            'message' => __('main.payment_redirect')
        ];
    }

    private function initiatePayment(float $amount, string $courseTitle)
    {
        try {
            $paymentData = $this->preparePaymentData($amount, $courseTitle);
            $response = $this->paymentGateway->sendPayment($paymentData);
            return $this->handlePaymentResponse($response);
        } catch (\Exception $e) {
            throw new \Exception('An error occurred while initiating the payment: ' . $e->getMessage());
        }
    }

    private function preparePaymentData(float $amount, string $courseTitle): array
    {
        return [
            "amount" => $amount * 100,
            "items"  => [
                [
                    "name"        => $courseTitle,
                    "amount"      => $amount * 100,
                    "description" => "Course Joining payment",
                    "quantity"    => 1
                ]
            ],
            "billing_data" => [
                "apartment"    => "NA",
                "first_name"   => 'name',
                "last_name"    => 'name',
                "phone_number" => '+201027388467',
                "email"        => 'diaam917@gmail.com',
                "country"      => 'EG',
                "street"       => "NA",
                "building"     => "NA",
                "city"         => "NA",
                "floor"        => "NA",
                "state"        => "NA"
            ]
        ];
    }

    private function handlePaymentResponse(array $response): \Illuminate\Http\JsonResponse
    {
        if ($response['success']) {
            return response()->json([
                'success'      => true,
                'message'      => 'Payment initiated successfully.',
                'redirect_url' => $response['redirect_url'] ?? null,
                'data'         => $response['data']
            ]);
        }

        return response()->json([
            'success' => false,
            'message' => $response['message'] ?? 'Payment initiation failed.',
            'data'    => $response['data'] ?? null
        ], 400);
    }
}
