<?php

namespace App\Services;

use LaravelDaily\Invoices\Invoice;
use LaravelDaily\Invoices\Classes\Party;
use LaravelDaily\Invoices\Classes\InvoiceItem;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class GenerateInvoice
{
    protected ?Party $customer = null;
    protected ?Party $seller = null;
    protected array $items = [];
    protected ?int $invoiceNumber = null;
    protected ?string $filename = null;
    protected bool $saveToDisk = false;
    protected string $disk = 'local';
    protected string $path = 'invoices';
    protected array $notes = [];
    protected ?string $logoPath = null;
    protected string $template = 'default';
    protected string $currencySymbol = '$';
    protected string $currencyCode = 'USD';
    protected string $dateFormat = 'm/d/Y';
    protected string $serialNumberFormat = '{SEQUENCE}/{YEAR}';

    /**
     * Set customer information
     */
    public function setCustomer(array $data): self
    {
        $this->customer = new Party($data);
        return $this;
    }

    /**
     * Set seller information
     */
    public function setSeller(array $data): self
    {
        $this->seller = new Party($data);
        return $this;
    }

    /**
     * Add an item to the invoice
     */
    public function addItem(
        string $title,
        float $price,
        int $quantity = 1,
        float $discount = 0,
        string $description = null
    ): self {
        $item = (new InvoiceItem())
            ->title($title)
            ->pricePerUnit($price)
            ->quantity($quantity)
            ->discount($discount);

        if ($description) {
            $item->description($description);
        }

        $this->items[] = $item;
        return $this;
    }

    /**
     * Add multiple items at once
     */
    public function addItems(array $items): self
    {
        foreach ($items as $item) {
            $this->addItem(
                $item['title'],
                $item['price'],
                $item['quantity'] ?? 1,
                $item['discount'] ?? 0,
                $item['description'] ?? null
            );
        }
        return $this;
    }

    /**
     * Set invoice number and sequence
     */
    public function setInvoiceNumber(int $number): self
    {
        $this->invoiceNumber = $number;
        return $this;
    }

    /**
     * Set custom filename
     */
    public function setFilename(string $filename): self
    {
        $this->filename = $filename;
        return $this;
    }

    /**
     * Configure to save invoice to disk
     */
    public function saveToDisk(string $disk = 'local', string $path = 'invoices'): self
    {
        $this->saveToDisk = true;
        $this->disk = $disk;
        $this->path = $path;
        return $this;
    }

    /**
     * Add notes to the invoice
     */
    public function addNotes(array $notes): self
    {
        $this->notes = array_merge($this->notes, $notes);
        return $this;
    }

    /**
     * Set company logo path
     */
    public function setLogo(string $path): self
    {
        $this->logoPath = $path;
        return $this;
    }

    /**
     * Set template
     */
    public function setTemplate(string $template): self
    {
        $this->template = $template;
        return $this;
    }

    /**
     * Set currency configuration
     */
    public function setCurrency(string $symbol, string $code): self
    {
        $this->currencySymbol = $symbol;
        $this->currencyCode = $code;
        return $this;
    }

    /**
     * Set date format
     */
    public function setDateFormat(string $format): self
    {
        $this->dateFormat = $format;
        return $this;
    }

    /**
     * Set serial number format
     */
    public function setSerialNumberFormat(string $format): self
    {
        $this->serialNumberFormat = $format;
        return $this;
    }

    /**
     * Generate and return the invoice
     */
    public function generate(): Invoice
    {
        $this->validateData();

        $invoice = Invoice::make()
            ->template($this->template)
            ->seller($this->seller)
            ->buyer($this->customer)
            ->sequence($this->invoiceNumber ?? rand(1000, 9999))
            ->serialNumberFormat($this->serialNumberFormat)
            ->date(now())
            ->dateFormat($this->dateFormat)
            ->currencySymbol($this->currencySymbol)
            ->currencyCode($this->currencyCode)
            ->addItems($this->items)
            ->filename($this->getFilename());

        if ($this->logoPath) {
            $invoice->logo($this->logoPath);
        }

        // if (!empty($this->notes)) {
        //     $invoice->notes($this->notes);
        // }

        return $invoice;
    }

    /**
     * Generate and download the invoice
     */
    public function download()
    {
        return $this->generate()->download();
    }

    /**
     * Generate and stream the invoice to browser
     */
    public function stream()
    {
        return $this->generate()->stream();
    }

    /**
     * Generate and save the invoice, returning the path
     */
    public function save(): string
    {
        $invoice = $this->generate();
        $path = $this->getFullPath();

        Storage::disk($this->disk)->put($path, $invoice->stream());

        return $path;
    }

    /**
     * Generate, save and return the full public URL
     */
    public function saveAndGetUrl(): string
    {
        $path = $this->save();
        return Storage::disk($this->disk)->url($path);
    }

    protected function validateData(): void
    {
        if (!$this->customer) {
            throw new \Exception('Customer information is required to generate an invoice');
        }

        if (!$this->seller) {
            throw new \Exception('Seller information is required to generate an invoice');
        }

        if (empty($this->items)) {
            throw new \Exception('At least one item is required to generate an invoice');
        }
    }

    protected function getFilename(): string
    {
        return $this->filename ?? 'invoice_' . Str::random(8) . '_' . now()->format('Ymd_His');
    }

    protected function getFullPath(): string
    {
        return rtrim($this->path, '/') . '/' . ltrim($this->getFilename(), '/') . '.pdf';
    }
}
