<?php

namespace App\Services\Notification;

use App\Models\User;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Kreait\Firebase\Factory;
use Kreait\Firebase\Messaging\CloudMessage;
use Kreait\Firebase\Messaging\Notification as FirebaseNotification;

class FirebaseNotificationService
{
    protected $messaging;

    public function __construct()
    {
        if (env('FIREBASE_CREDENTIALS')) {
        $factory = (new Factory)
            ->withServiceAccount(config('services.firebase.credentials'));
        $this->messaging = $factory->createMessaging();
        }
        else{
            // $this->messaging = null;
        }
    }

    public function sendToDevice($token, array $data)
    {
        $notification = FirebaseNotification::create($data['title'], $data['content']);

        $message = CloudMessage::withTarget('token', $token)
            ->withNotification($notification)
            ->withData($data);

        return $this->messaging->send($message);
    }

    public function sendToMultipleDevices(array $tokens, array $data)
    {
        try {
            Log::info('Preparing to send Firebase notifications to multiple devices', [
                'token_count' => count($tokens),
                'data' => $data
            ]);

            $notification = FirebaseNotification::create(
                $data['title'],
                $data['content'],
            );

            // $notification=FirebaseNotification::create([
            //     'title' => $data['title'],
            //     'content' => $data['content'],
            //     'some_column' => null,
            //     'name' => $data['type'],
            // ]);


            $message = CloudMessage::new()
                ->withNotification($notification)
                ->withData($data);

            $result = $this->messaging->sendMulticast($message, $tokens);

            Log::info('Firebase multicast notification completed', [
                'success_count' => $result->successes()->count(),
                'failure_count' => $result->failures()->count()
            ]);

            if ($result->failures()->count() > 0) {
                Log::warning('Some Firebase notifications failed', [
                    'failures' => $result->failures()->getItems()
                ]);
            }

            return $result;
        } catch (\Exception $e) {
            Log::error('Firebase multicast notification failed', [
                'token_count' => count($tokens),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return null;
        }
    }

public function sendToMultipleDevicesUrl(array $tokens, array $data)
{
    try {
        Log::info('Preparing Firebase notifications', [
            'token_count' => count($tokens),
            'data' => $data
        ]);

        $pushResult = null;
        if (!empty($tokens)) {
            $notification = FirebaseNotification::create($data['title'], $data['content']);
            $message = CloudMessage::new()
                ->withNotification($notification)
                ->withData($data);

            $pushResult = $this->messaging->sendMulticast($message, $tokens);

            Log::info('FCM multicast results', [
                'success_count' => $pushResult->successes()->count(),
                'failure_count' => $pushResult->failures()->count()
            ]);
        }

        $this->saveToFirebaseRealtimeDB($data);

        return $pushResult;

    } catch (\Exception $e) {
        Log::error('Firebase operation failed', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        return null;
    }
}

    protected function saveToFirebaseRealtimeDB(array $data)
    {
        try {
            $factory = (new Factory)
                ->withServiceAccount(config('services.firebase.credentials'))
                ->withDatabaseUri(config('services.firebase.database_url'));

            $database = $factory->createDatabase();

            $notificationRef = $database->getReference('notifications')->push();

            $notificationData = [
                'title'       => $data['title'],
                'content'     => $data['content'],
                'type'        => $data['type'] ?? 'general',
                'created_at'  => time(),
                'url'         => $data['url'] ?? null,
                'is_read'     => false
            ];

            $notificationRef->set($notificationData);

            Log::info('Notification saved to Firebase Realtime DB', [
                'notification_id' => $notificationRef->getKey()
            ]);

            return true;
        } catch (\Exception $e) {
            Log::error('Failed to save notification to Firebase Realtime DB', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return false;
        }
    }

    public function sendToTopic($topic, array $data)
    {
        $notification = FirebaseNotification::create($data['title'], $data['content']);

        $message = CloudMessage::withTarget('topic', $topic)
            ->withNotification($notification)
            ->withData($data);

        return $this->messaging->send($message);
    }


    protected function getFirebaseDatabase()
    {
        return (new Factory)
            ->withServiceAccount(config('services.firebase.credentials'))
            ->withDatabaseUri(config('services.firebase.database_url'))
            ->createDatabase();
    }
    public function sendToUsersViaFirebase(array $userIds, array $notificationData): bool
    {
        try {
            if (empty($userIds)) {
                Log::warning('No user IDs provided for Firebase notification');
                return false;
            }

            $database = $this->getFirebaseDatabase();
            $successCount = 0;

            foreach ($userIds as $userId) {
                try {
                    $user=User::find($userId);
                    // Create user-specific notification path
                    $userNotificationsRef = $database->getReference("user_notifications/$userId")->push();

                    $notificationPayload = [
                        'id' => Str::uuid()->toString(),
                        'user_id' => $userId,
                        'title' => $notificationData['title'] ?? 'New Notification',
                        'content' => $this->replaceVariables($notificationData['content'], $user) ?? '',
                        'data' => $notificationData['data'] ?? [],
                        'created_at' => time(),
                        'is_read' => false,
                        'action' => $notificationData['action'] ?? null,
                        'action_id' => $notificationData['action_id'] ?? null
                    ];

                    $userNotificationsRef->set($notificationPayload);
                    $successCount++;

                    Log::debug('Notification saved to Firebase', [
                        'user_id' => $userId,
                        'notification_id' => $userNotificationsRef->getKey()
                    ]);

                } catch (\Exception $e) {
                    Log::error('Failed to save notification for user', [
                        'user_id' => $userId,
                        'error' => $e->getMessage()
                    ]);
                    continue;
                }
            }

            Log::info('Firebase notifications completed', [
                'total_users' => count($userIds),
                'successful' => $successCount,
                'failed' => count($userIds) - $successCount
            ]);

            return $successCount > 0;

        } catch (\Exception $e) {
            Log::error('Firebase operation failed', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            return false;
        }
    }


    protected function buildUserCondition(array $userIds): string
    {
        if (empty($userIds)) {
            throw new \InvalidArgumentException('User IDs cannot be empty');
        }

        $conditions = [];
        foreach ($userIds as $userId) {
            $safeUserId = preg_replace('/[^a-zA-Z0-9_-]/', '', $userId);
            $conditions[] = "'user_{$safeUserId}' in topics";
        }

        return implode(' || ', $conditions);
    }

    public function replaceVariables($message, $user, array $additionalVars = [])
    {
        $replacements = [
            '{{name}}' => $user->name ?? '',
            '{{email}}' => $user->email ?? '',
            '{{phone}}' => $user->phone ?? '',
            '{{gender}}' => ($user->gender ?? '') != 'male' ? 'ة' : '',
        ];

        foreach ($additionalVars as $key => $value) {
            $replacements['{{'.$key.'}}'] = $value;
        }

        return str_replace(array_keys($replacements), array_values($replacements), $message);
    }
}
