<?php

namespace App\Services;

use App\Enums\TicketStatus;
use App\Enums\TicketType;
use App\Http\Traits\MediaHandler;
use App\Models\Ticket;
use App\Models\TicketMessage;

class TicketService
{
    use MediaHandler;

    /**
     * Handle creating a message for a ticket.
     *
     * @param Ticket $ticket
     * @param array $data
     * @return TicketMessage
     */
    public function addMessage(Ticket $ticket, array $data)
    {
        $messageData = [
            'ticket_id' => $ticket->id,
            'type' => isset($data['content_file']) ? TicketType::AUDIO->value : TicketType::TEXT->value,
            'content' => isset($data['content_file']) ? $this->upload($data['content_file'], 'tickets') : $data['content'],
            'created_by' => auth()->id(),
        ];

        $message = TicketMessage::create($messageData);
        return $message->load('ticket.student:id,name,email');
    }

    /**
     * Assign an admin to a ticket if it is still pending.
     *
     * @param Ticket $ticket
     * @param int $adminId
     */
    public function assignAdmin(Ticket $ticket, int $adminId)
    {
        if ($ticket->status === TicketStatus::PENDING) {
            $ticket->update([
                'status' => TicketStatus::OPENED,
                'admin_id' => $adminId,
            ]);
        }
    }

    /**
     * Create a new ticket and add an initial message.
     *
     * @param array $data The validated ticket data.
     * @return Ticket The created ticket with related student and messages loaded.
     */
    public function createTicket(array $data): Ticket
    {
        $ticketData = [
            'description' => $data['description'],
            'user_id' => auth()->id(),
            'status' => TicketStatus::PENDING->value,
        ];

        $ticket = Ticket::create($ticketData);

        return $ticket->load('student:id,name,email', 'messages');
    }
}
