<?php

namespace Database\Seeders;

use App\Models\Course;
use App\Models\Category;
use App\Models\Round;
use App\Models\Lecture;
use App\Enums\DefineStatus;
use Illuminate\Database\Seeder;

class ComprehensiveTestSeeder extends Seeder
{
    public function run(): void
    {
        $this->seedCategories();

        $courses = $this->seedCourses();

        foreach ($courses as $course) {
            $rounds = $this->seedRounds($course['id']);

            foreach ($rounds as $round) {
                $this->seedLectures($round['id']);
            }
        }
    }

    protected function seedCategories(): void
    {
        $levels = [
            ['name' => 'A0/A1', 'order' => 1],
            ['name' => 'A2', 'order' => 2],
            ['name' => 'B1', 'order' => 3],
            ['name' => 'B2', 'order' => 4],
            ['name' => 'C1', 'order' => 5],
            ['name' => 'C2', 'order' => 6],
        ];

        foreach ($levels as $level) {
            Category::firstOrCreate(
                ['name' => $level['name']],
                [
                    'order' => $level['order'],
                    'status' => DefineStatus::ACTIVE->value
                ]
            );
        }
    }

    protected function seedCourses(): array
    {
        $courses = [
            [
                'category_id'             => Category::where('name', 'A0/A1')->first()->id,
                'title'                   => 'Course A0/A1',
                'image'                   => 'https://placehold.co/800x600/png?text=English+Vision+A0/A1',
                'price'                   => 4500,
                'description'             => 'Beginner English course for absolute starters',
                'preview_video'           => 'https://youtu.be/ZZHVKw-djZ0?si=3rkBQqbyKl1fBms1',
                'preview_video_thumbnail' => 'https://img.youtube.com/vi/ZZHVKw-djZ0/sddefault.jpg',
                'status'                  => DefineStatus::ACTIVE->value,
                'sessions_count'          => 24,
                'duration'                => 120,
                'what_you_will_learn'     => json_encode([
                    'Learn basic English vocabulary',
                    'Master simple grammar structures',
                    'Develop elementary conversation skills',
                    'Understand common everyday phrases',
                    'Build a foundation for further language learning'
                ]),
                'learning_plan' => 'https://www.ulethbridge.ca/sites/default/files/docs/LearningPlanSample.pdf',
            ],
            [
                'category_id'             => Category::where('name', 'A2')->first()->id,
                'title'                   => 'Course A2',
                'image'                   => 'https://placehold.co/800x600/png?text=English+Vision+A2',
                'price'                   => 5000,
                'description'             => 'Elementary English course',
                'preview_video'           => 'https://youtu.be/ZZHVKw-djZ0?si=3rkBQqbyKl1fBms1',
                'preview_video_thumbnail' => 'https://img.youtube.com/vi/ZZHVKw-djZ0/sddefault.jpg',
                'status'                  => DefineStatus::ACTIVE->value,
                'sessions_count'          => 30,
                'duration'                => 150,
                'what_you_will_learn'     => json_encode([
                    'Expand your English vocabulary',
                    'Improve grammar understanding',
                    'Enhance conversation abilities',
                    'Practice everyday communication',
                    'Develop reading and writing skills'
                ]),
                'learning_plan' => 'https://www.ulethbridge.ca/sites/default/files/docs/LearningPlanSample.pdf',
            ],
            [
                'category_id'             => Category::where('name', 'B1')->first()->id,
                'title'                   => 'Course B1',
                'image'                   => 'https://placehold.co/800x600/png?text=English+Vision+B1',
                'price'                   => 5500,
                'description'             => 'Intermediate English course',
                'preview_video'           => 'https://youtu.be/ZZHVKw-djZ0?si=3rkBQqbyKl1fBms1',
                'preview_video_thumbnail' => 'https://img.youtube.com/vi/ZZHVKw-djZ0/sddefault.jpg',
                'status'                  => DefineStatus::ACTIVE->value,
                'sessions_count'          => 36,
                'duration'                => 180,
                'what_you_will_learn'     => json_encode([
                    'Master intermediate vocabulary',
                    'Apply complex grammar structures',
                    'Engage in fluent conversations',
                    'Understand native speakers',
                    'Express opinions on various topics'
                ]),
                'learning_plan' => 'https://www.ulethbridge.ca/sites/default/files/docs/LearningPlanSample.pdf',
            ],
            [
                'category_id'             => Category::where('name', 'B2')->first()->id,
                'title'                   => 'Course B2',
                'image'                   => 'https://placehold.co/800x600/png?text=English+Vision+B2',
                'price'                   => 6000,
                'description'             => 'Upper-intermediate English course',
                'preview_video'           => 'https://youtu.be/ZZHVKw-djZ0?si=3rkBQqbyKl1fBms1',
                'preview_video_thumbnail' => 'https://img.youtube.com/vi/ZZHVKw-djZ0/sddefault.jpg',
                'status'                  => DefineStatus::ACTIVE->value,
                'sessions_count'          => 42,
                'duration'                => 210,
                'what_you_will_learn'     => json_encode([
                    'Master advanced vocabulary',
                    'Apply sophisticated grammar',
                    'Engage in complex conversations',
                    'Understand cultural nuances',
                    'Express nuanced opinions'
                ]),
                'learning_plan' => 'https://www.ulethbridge.ca/sites/default/files/docs/LearningPlanSample.pdf',
            ],
            [
                'category_id'             => Category::where('name', 'C1')->first()->id,
                'title'                   => 'Course C1',
                'image'                   => 'https://placehold.co/800x600/png?text=English+Vision+C1',
                'price'                   => 6500,
                'description'             => 'Advanced English course',
                'preview_video'           => 'https://youtu.be/ZZHVKw-djZ0?si=3rkBQqbyKl1fBms1',
                'preview_video_thumbnail' => 'https://img.youtube.com/vi/ZZHVKw-djZ0/sddefault.jpg',
                'status'                  => DefineStatus::ACTIVE->value,
                'sessions_count'          => 48,
                'duration'                => 240,
                'what_you_will_learn'     => json_encode([
                    'Master professional vocabulary',
                    'Apply advanced grammar in context',
                    'Participate in academic discussions',
                    'Understand complex texts',
                    'Write formal documents'
                ]),
                'learning_plan' => 'https://www.ulethbridge.ca/sites/default/files/docs/LearningPlanSample.pdf',
            ],
            [
                'category_id'             => Category::where('name', 'C2')->first()->id,
                'title'                   => 'Course C2',
                'image'                   => 'https://placehold.co/800x600/png?text=English+Vision+C2',
                'price'                   => 7000,
                'description'             => 'Proficiency English course',
                'preview_video'           => 'https://youtu.be/ZZHVKw-djZ0?si=3rkBQqbyKl1fBms1',
                'preview_video_thumbnail' => 'https://img.youtube.com/vi/ZZHVKw-djZ0/sddefault.jpg',
                'status'                  => DefineStatus::ACTIVE->value,
                'sessions_count'          => 54,
                'duration'                => 270,
                'what_you_will_learn'     => json_encode([
                    'Master nuanced vocabulary',
                    'Apply grammar with native-like fluency',
                    'Engage in high-level debates',
                    'Understand subtle cultural references',
                    'Write with professional precision'
                ]),
                'learning_plan' => 'https://www.ulethbridge.ca/sites/default/files/docs/LearningPlanSample.pdf',
            ],
        ];

        Course::insert($courses);
        return Course::all()->toArray();
    }

    protected function seedRounds(int $courseId): array
    {
        $rounds = [
            [
                'course_id'     => $courseId,
                'instructor_id' => 1,
                'name'          => 'Round 1',
                'start_date'    => now()->addDays(10)->format('Y-m-d'),
                'end_date'      => now()->addDays(10 + 30)->format('Y-m-d'),
                'discount'      => 10,
            ],
            [
                'course_id'     => $courseId,
                'instructor_id' => 2,
                'name'          => 'Round 2',
                'start_date'    => now()->addDays(45)->format('Y-m-d'),
                'end_date'      => now()->addDays(45 + 30)->format('Y-m-d'),
                'discount'      => 0,
            ],
            [
                'course_id'     => $courseId,
                'instructor_id' => 1,
                'name'          => 'Round 3',
                'start_date'    => now()->addDays(80)->format('Y-m-d'),
                'end_date'      => now()->addDays(80 + 30)->format('Y-m-d'),
                'discount'      => 15,
            ],
        ];

        Round::insert($rounds);
        return Round::where('course_id', $courseId)->get()->toArray();
    }

    protected function seedLectures(int $roundId): void
    {
        $lectures = [];
        $startDate = Round::find($roundId)->start_date;
        $attachments = [];

        for ($week = 1; $week <= 4; $week++) {
            for ($session = 1; $session <= 2; $session++) {
                $lecture = Lecture::create([
                    'round_id'    => $roundId,
                    'name'        => "Week $week - Session $session",
                    'url'         => 'https://www.youtube.com/watch?v=ZZHVKw-djZ0',
                    'start_time'  => "$startDate " . ($week === 1 ? '10:00:00' : '14:00:00'),
                    'status'      => 1,
                    'week_number' => $week,
                ]);

                $lecture->attachments()->createMany([
                    [
                        'name' => "Lecture Notes - Week $week Session $session",
                        'path' => "invoices/invoice_4lu1SCD7_20250529_211155.pdf",
                        'attachable_type' => Lecture::class,
                    ],
                    [
                        'name' => "Exercise Sheet - Week $week Session $session",
                        'path' => "invoices/invoice_4lu1SCD7_20250529_211155.pdf",
                        'attachable_type' => Lecture::class,
                    ]
                ]);
            }
            $startDate = date('Y-m-d', strtotime($startDate . ' +7 days'));
        }
    }
}
